<?php

namespace Modules\Amazon\Managers;

/*
 * The purpose of this class is to manage the process of inventory for the Amazon FBA Warehouse
 *
 * This class will:
 *
 * Request a schedule of relevant reports
 * Process reports oldest to newest, positive quantity first then negative quantity (Amazon doesn't support backorders)
 * Create inventory movements linked to appropriate Amazon reports
 */

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelManager;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Repositories\AmazonReportRepository;
use Modules\Amazon\Services\AmazonClient;
use Throwable;

class AmazonFbaInventoryManager extends AbstractSalesChannelManager
{
    private AmazonReportRepository $amazonReportRepository;

    public function __construct(private readonly AmazonIntegrationInstance $amazonIntegrationInstance)
    {
        $this->amazonReportRepository = app(AmazonReportRepository::class);
        parent::__construct($amazonIntegrationInstance, new AmazonClient($amazonIntegrationInstance));
    }

    /**
     * @throws Throwable
     */
    public function requestLedgerReport()
    {
        /*
         * FbaReportInventoryLedger has everything
         *
         * Can query the max(date) from this model to determine what new requests need to be made (for what start date)
         */

        $lastUpdatedAfter = $this->amazonReportRepository->getLedgerReportLastUpdatedAfter($this->amazonIntegrationInstance);

        // Request Report
        (new AmazonReportManager($this->amazonIntegrationInstance))->requestCreateSync(
            AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER,
            $lastUpdatedAfter,
        );
    }

    /*
     *
     * First step or processing is to make sure all reports are generated to cover the same dates and that each
     * of those reports have their data saved in appropriate tables, but especially the FbaReportInventoryLedger
     * since that is what we'll be using for inventory integrity.
     *
     * Can group FbaReportInventoryLedger records by date, then by a new enum for
     * FbaReportInventoryLedgerEventTypeEnum.  Within each group if more than 1 record, order by defined order
     *  (see https://lucid.app/lucidchart/d4aec1c4-6267-46ce-875a-29e33fe16f82/edit?page=0_0&invitationId=inv_3b5b78c4-d209-4156-b21e-187de99d57a3#)
     *
     *
     * Processing of FbaReportInventoryLedger records in the correct order will then use the is_inventory_processed
     * flag to track the creation of inventory movements.
     *
     * Alternatively to is_inventory_processed, we could use the relationship to inventory movements.  The relationship
     * could be just between inventory movements and FbaReportInventoryLedger.
     *
     * There may need to be an additional relationship to establish between detail reports (returns/shipments/removals)
     * and the corresponding inventory ledger.  Extra detail in the detail reports may have future user in reporting
     * or financials
     *
     *
     *
     * TODO: We need to be able to link Amazon Fba Report Inventory Ledger to something like Amazon Fba Report Shipments
     *  We have no have no way to reliably automate this.  We can maybe automate it in the simplest cases, and require
     *  User intervention is vague cases.  The benefit of connecting  these elements is if we connect
     *  Amazon Fba Report Inventory Ledger <=> Amazon Fba Report Shipments <=> SalesOrderFulfillmentLines, we can:
     *  a) Perform checks and balances to make sure that Amazon Inventory is all accounted for
     *  b) Have the ability for User to see inventory movements for FBA Sales Orders via UI
     *  c) Show extra metadata to the User such as tracking #
     *  ---
     *  The link is critical for financial reporting.  For example, if we don't have the ability to link an fba
     *  sales order line to a fifo layer, we can't establish a cost
     */
}
