<?php

namespace Modules\Amazon\Jobs;

use App\Exceptions\ApiException;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Modules\Amazon\Data\AmazonResponseData;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Jobs\Abstractions\AmazonJob;
use Modules\Amazon\Managers\AmazonReportManager;
use Modules\Amazon\Repositories\AmazonIntegrationInstanceRepository;
use Modules\Amazon\Repositories\AmazonReportRepository;
use Throwable;

class SyncAmazonReportsJob extends AmazonJob implements ShouldBeUnique
{
    protected AmazonReportRepository $reportRepository;

    protected AmazonIntegrationInstanceRepository $amazonInstanceRepository;

    public int $uniqueFor = 60 * 5;

    public function uniqueId(): string
    {
        return $this->amazonIntegrationInstance->id;
    }

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected AmazonIntegrationInstance $amazonIntegrationInstance)
    {
        $this->reportRepository = app(AmazonReportRepository::class);
        $this->amazonInstanceRepository = app(AmazonIntegrationInstanceRepository::class);
    }

    /**
     * Get reports.
     *
     * @throws GuzzleException
     * @throws ApiException
     * @throws Throwable
     */
    public function apiCall(): AmazonResponseData
    {
        $this->reportRepository
            ->getInitializedReports($this->amazonIntegrationInstance)
            ->each(function ($amazonReport) {
                //Log::debug('Syncing report ' . $amazonReport->reportType->value . ' ' . $amazonReport->reportId . ' for ' . $this->amazonIntegrationInstance->name);
                (new AmazonReportManager($this->amazonIntegrationInstance))->sync($amazonReport);
            });

        return AmazonResponseData::from([
            'collection' => collect([]),
            'nextToken' => null,
        ]);
    }
}
