<?php

namespace Modules\Amazon\Http\Controllers;

use App\Data\IdSelectionData;
use App\DataTable\DataTableResource;
use App\Response;
use Exception;
use Modules\Amazon\Entities\AmazonFbaInboundShipment;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonNewFbaInboundShipment;
use Modules\Amazon\Http\Controllers\Concerns\AmazonExportableControllerTrait;
use Modules\Amazon\Http\Resources\AmazonFbaInboundShipmentResource;
use Modules\Amazon\Http\Resources\AmazonNewFbaInboundShipmentResource;
use Modules\Amazon\Managers\AmazonNewInboundManager;
use Modules\Amazon\Repositories\AmazonNewFbaInboundShipmentRepository;

class AmazonNewFbaInboundShipmentController extends AbstractAmazonDataTableController
{
    use AmazonExportableControllerTrait;

    private AmazonNewFbaInboundShipmentRepository $shipments;

    public function __construct()
    {
        $this->shipments = app(AmazonNewFbaInboundShipmentRepository::class);
        parent::__construct();
    }

    protected function getResource(): DataTableResource|string
    {
        return AmazonNewFbaInboundShipmentResource::class;
    }

    protected function getModel(): string
    {
        return AmazonNewFbaInboundShipment::class;
    }

    public function show(
        AmazonIntegrationInstance $integrationInstance,
        AmazonNewFbaInboundShipment $shipment
    ): ?Response {
        $shipment->load([
            'amazonFbaInboundShipmentItems.amazonProduct',
            'skuLink',
            'ledgers'
        ]);

        return $this->response->addData(AmazonNewFbaInboundShipmentResource::make($shipment));
    }

    /**
     * @throws Exception
     */
    public function process(IdSelectionData $data, AmazonIntegrationInstance $integrationInstance): Response
    {
        $processedShipments = (new AmazonNewInboundManager($integrationInstance))->process($data->ids);
        return $this->response->setMessage('Inbound shipments processed')->addData($this->getResource()::collection($processedShipments));
    }

    /**
     * @throws Exception
     */
    public function processAll(AmazonIntegrationInstance $integrationInstance): Response
    {
        (new AmazonNewInboundManager($integrationInstance))->processUnprocessed();
        return $this->response->setMessage('Inbound shipments processed');
    }

    /**
     * @throws Exception
     */
    public function unprocess(IdSelectionData $data, AmazonIntegrationInstance $integrationInstance): Response
    {
        (new AmazonNewInboundManager($integrationInstance))->unprocess($data->ids);
        return $this->response->setMessage('Inbound shipments unprocessed');
    }

    public function setSentBeforeInitialCount(IdSelectionData $data): Response
    {
        $this->shipments->setValueForIds($data->ids, AmazonNewFbaInboundShipment::class, [
            'is_before_initial_count' => $data->value == 'true',
            'errorLog' => null,
        ]);

        return $this->response->setMessage('Inbound shipments updated');
    }

    public function getUnlinkedShipments(AmazonIntegrationInstance $integrationInstance)
    {
        $amazonFbaInboundShipments = $this->shipments->getUnlinkedShipments($integrationInstance);

        return $this->response->addData($amazonFbaInboundShipments);
    }
}
