<?php

namespace Modules\Amazon\Http\Controllers;

use App\DataTable\DataTableResource;
use App\Response;
use Exception;
use Modules\Amazon\Data\LinkAmazonFnskuProductToSkuProductData;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Exceptions\AmazonProductsUnmappedException;
use Modules\Amazon\Http\Controllers\Concerns\AmazonExportableControllerTrait;
use Modules\Amazon\Http\Resources\AmazonFnskuProductResource;
use Modules\Amazon\Jobs\AmazonInitializeSkuProductsJob;
use Modules\Amazon\Managers\AmazonFnskuProductManager;
use Modules\Amazon\Repositories\AmazonReportRepository;
use Throwable;

class AmazonFnskuProductController extends AbstractAmazonDataTableController
{
    use AmazonExportableControllerTrait;

    private AmazonReportRepository $amazonReportRepository;

    public function __construct()
    {
        $this->amazonReportRepository = app(AmazonReportRepository::class);
        parent::__construct();
    }

    protected function getModel(): string
    {
        return AmazonFnskuProduct::class;
    }

    protected function getResource(): DataTableResource|string
    {
        return AmazonFnskuProductResource::class;
    }

    public function show(AmazonIntegrationInstance $integrationInstance, AmazonFnskuProduct $fnskuProduct): ?Response
    {
        $fnskuProduct->load([
            'product',
            'amazonFbaReportInventoryLedgers' => fn ($query) => $query->select(
                'id',
                'integration_instance_id',
                'fnsku',
                'msku',
                'reference_id',
                'country',
                'disposition',
                'date',
                'event_type',
                'quantity',
                'errorLog',
                'sku_link_id',
                'sku_link_type',
                'reconciled_at',
            )->reconcileSort(),
            'amazonFbaInitialInventory' => fn ($query) => $query->select(
                'id',
                'integration_instance_id',
                'fnsku',
                'date',
                'msku',
                'in_transit_between_warehouses',
                'ending_warehouse_balance',
                'total_inventory_quantity',
                'location',
                'disposition',
                'fifo_layer_id',
            ),
        ]);

        return $this->response->addData($this->getResource()::make($fnskuProduct));
    }

    /**
     * @throws Throwable
     */
    public function generateFnskuProducts(AmazonIntegrationInstance $integrationInstance): Response
    {
        try {
            $created = (new AmazonFnskuProductManager($integrationInstance))->generateFnskuProducts();
            dispatch(new AmazonInitializeSkuProductsJob($integrationInstance));
        } catch(AmazonProductsUnmappedException $e) {
            return $this->response->setMessage($e->getMessage())->setStatus(400);
        }
        return $this->response->setMessage($created . ' FNSKU products generated');
    }

    /**
     * @throws Exception
     */
    public function linkToProduct(
        LinkAmazonFnskuProductToSkuProductData $data,
        AmazonIntegrationInstance $integrationInstance,
    ): Response
    {
        $amazonFnskuProduct = (new AmazonFnskuProductManager($integrationInstance))->linkFnskuProductToSkuProduct($data);
        return $this->response->addData(AmazonFnskuProductResource::make($amazonFnskuProduct));
    }
}
