<?php

namespace Modules\Amazon\Http\Controllers;

use App\DataTable\DataTable;
use App\DataTable\DataTableResource;
use App\Http\Controllers\Controller;
use App\Response;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Validation\Rule;
use Modules\Amazon\Entities\AmazonFinancialEventGroup;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Http\Resources\AmazonFinancialEventGroupResource;
use Modules\Amazon\Jobs\CreateAmazonFinancialEventGroupAccountingTransactionsJob;
use Modules\Amazon\Managers\AmazonFinanceManager;
use Modules\Amazon\Repositories\AmazonIntegrationInstanceRepository;

class AmazonFinancialEventGroupController extends AbstractAmazonDataTableController
{
    protected function getResource(): DataTableResource|string
    {
        return AmazonFinancialEventGroupResource::class;
    }

    protected function getModel(): string
    {
        return AmazonFinancialEventGroup::class;
    }

    public function show(AmazonIntegrationInstance $integrationInstance, AmazonFinancialEventGroup $amazonFinancialEventGroup): ?Response
    {
        $amazonFinancialEventGroup->load('accountingTransaction');

        return $this->response->addData(AmazonFinancialEventGroupResource::make($amazonFinancialEventGroup));
    }

    public function createAccountingTransactions(
        Request $request,
        AmazonIntegrationInstance $integrationInstance,
    ): Response|AmazonFinancialEventGroup {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => [
                'integer',
                Rule::exists('amazon_financial_event_groups', 'id'),
            ],
        ]);

        // We want this to happen instantly if there is just 1 accounting transaction to create
        if (count($request->get('ids')) === 1) {
            $this->dispatchSync(new CreateAmazonFinancialEventGroupAccountingTransactionsJob(
                $integrationInstance,
                $request->get('ids')
            ));

            /** @var AmazonFinancialEventGroup $amazonFinancialEventGroup */
            $amazonFinancialEventGroup = AmazonFinancialEventGroup::with('accountingTransaction')->findOrFail($request->get('ids')[0]);

            return $amazonFinancialEventGroup;
        }

        dispatch(new CreateAmazonFinancialEventGroupAccountingTransactionsJob(
            $integrationInstance,
            $request->get('ids')
        ));

        return $this->response->success()->setMessage(__('messages.success.queued'));
    }
}
