<?php

namespace Modules\Amazon\Http\Controllers;

use App\Data\IdSelectionData;
use App\DataTable\DataTableResource;
use App\Exceptions\PurchaseOrder\NotOpenPurchaseOrderException;
use App\Exceptions\WarehouseTransfers\NotOpenWarehouseTransferException;
use App\Response;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Enums\Entities\FbaInventoryLedgerReportEventTypeEnum;
use Modules\Amazon\Exceptions\LedgerEventTypeWithNoDetailReportException;
use Modules\Amazon\Exceptions\LedgerReconciledOutOfOrderException;
use Modules\Amazon\Exceptions\LedgerUnreconciledOutOfOrderException;
use Modules\Amazon\Http\Controllers\Concerns\AmazonExportableControllerTrait;
use Modules\Amazon\Http\Resources\AmazonFbaReportInventoryLedgerResource;
use Modules\Amazon\Managers\AmazonLedgerManager;
use Modules\Amazon\Repositories\AmazonLedgerRepository;
use Throwable;

class AmazonFbaLedgerController extends AbstractAmazonDataTableController
{
    use AmazonExportableControllerTrait;

    private AmazonLedgerRepository $ledgers;

    protected function getModel(): string
    {
        return AmazonFbaReportInventoryLedger::class;
    }

    protected function getResource(): DataTableResource|string
    {
        return AmazonFbaReportInventoryLedgerResource::class;
    }

    public function __construct()
    {
        $this->ledgers = app(AmazonLedgerRepository::class);
        parent::__construct();
    }

    public function show(AmazonIntegrationInstance $integrationInstance, AmazonFbaReportInventoryLedger $ledger): ?Response
    {
        $ledger->load([
            'ledgerDetails.detail',
            'amazonProduct',
            'amazonFnskuProduct',
        ]);

        return $this->response->addData(AmazonFbaReportInventoryLedgerResource::make($ledger));
    }

    /**
     * @throws Exception
     */
    public function getUnmatchedDetailReports(AmazonIntegrationInstance $integrationInstance, AmazonFbaReportInventoryLedger $amazonFbaReportInventoryLedger): Response
    {
        try {
            $detailReports = $this->ledgers->getUnmatchedDetailReportsForLedger($amazonFbaReportInventoryLedger);
        } catch (LedgerEventTypeWithNoDetailReportException) {
            return $this->response->addData([])->addAdditional('alert', 'Detail reports don\'t exist for this event type');
        }
        return $this->response->addData($detailReports);
    }

    /**
     * @throws Exception
     */
    public function getUnmatchedLedgers(Request $request): Response
    {
        $request->validate([
            'detailReportId' => 'required|integer',
            'eventType' => 'required|in:' . implode(',', FbaInventoryLedgerReportEventTypeEnum::HasDetailReports),
        ]);

        $eventType = FbaInventoryLedgerReportEventTypeEnum::from($request->get('eventType'));

        $ledgers = $this->ledgers->getUnmatchedLedgersForDetailReport($request->get('detailReportId'), $eventType);

        return $this->response->addData($ledgers);
    }

    /**
     * @throws Exception
     */
    public function linkToDetailReports(Request $request, AmazonIntegrationInstance $integrationInstance, AmazonFbaReportInventoryLedger $ledger)
    {
        $request->validate([
            'detailReportIds' => 'required|array'
        ]);

        $ledger = (new AmazonLedgerManager($integrationInstance))->linkLedgerToDetailReports($ledger, $request->get('detailReportIds'));

        return $this->response->addData(AmazonFbaReportInventoryLedgerResource::make($ledger));
    }

    /**
     * @throws Throwable
     */
    public function reconcileLedger(AmazonIntegrationInstance $integrationInstance, AmazonFbaReportInventoryLedger $ledger): Response
    {
        $ledgers = (new AmazonLedgerManager($integrationInstance))->reconcileLedgers(null, $ledger->amazonFnskuProduct, [$ledger->id]);
        $fnskuProduct = $ledgers->first()->amazonFnskuProduct->refresh();
        return $this->response->addData(AmazonFbaReportInventoryLedgerResource::collection($ledgers))->addAdditional('fnskuProductBalance', [
            'reconciled_quantity' => $fnskuProduct->reconciled_quantity,
        ]);
    }

    /**
     * @throws Throwable
     * @throws NotOpenPurchaseOrderException
     * @throws NotOpenWarehouseTransferException
     */
    public function reconcileLedgers(IdSelectionData $data, AmazonIntegrationInstance $integrationInstance): Response
    {
        (new AmazonLedgerManager($integrationInstance))->reconcileLedgers(null, null, $data->ids);
        return $this->response->setMessage('Ledgers successfully processed for reconciliation');
    }

    /**
     * @throws Throwable
     */
    public function reconcileFnskuLedgers(AmazonIntegrationInstance $integrationInstance, AmazonFnskuProduct $fnskuProduct): Response
    {
        $ledgers = (new AmazonLedgerManager($integrationInstance))->reconcileLedgers(null, $fnskuProduct);
        $fnskuProduct->refresh();
        return $this->response->addData(AmazonFbaReportInventoryLedgerResource::collection($ledgers))->addAdditional('fnskuProductBalance', [
            'reconciled_quantity' => $fnskuProduct->reconciled_quantity,
        ]);
    }

    /**
     * @throws Throwable
     */
    public function reconcileAll(AmazonIntegrationInstance $integrationInstance): Response
    {
        (new AmazonLedgerManager($integrationInstance))->reconcileLedgers();
        return $this->response->setMessage('Ledger reconciliation process completed');
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function unreconcileLedger(AmazonIntegrationInstance $integrationInstance, AmazonFbaReportInventoryLedger $ledger): Response
    {
        $ledgers = (new AmazonLedgerManager($integrationInstance))->unreconcileLedgers($ledger->amazonFnskuProduct, [$ledger->id]);
        $fnskuProduct = $ledgers->first()->amazonFnskuProduct->refresh();
        return $this->response->addData(AmazonFbaReportInventoryLedgerResource::collection($ledgers))->addAdditional('fnskuProductBalance', [
            'reconciled_quantity' => $fnskuProduct->reconciled_quantity,
        ]);
    }

    /**
     * @throws Exception
     */
    public function unreconcileLedgers(Request $request, AmazonIntegrationInstance $integrationInstance): Response
    {
        $request->validate([
            'ids.*' => 'required|exists:amazon_fba_report_inventory_ledger,id',
        ]);

        $ledgers = (new AmazonLedgerManager($integrationInstance))->unreconcileLedgers(null, $request->get('ids'));

        return $this->response->addData(AmazonFbaReportInventoryLedgerResource::collection($ledgers));
    }

    /**
     * @throws Throwable
     */
    public function unreconcileFnskuLedgers(AmazonIntegrationInstance $integrationInstance, AmazonFnskuProduct $fnskuProduct): Response
    {
        $ledgers = (new AmazonLedgerManager($integrationInstance))->unreconcileLedgers($fnskuProduct);
        $fnskuProduct->refresh();
        return $this->response->addData(AmazonFbaReportInventoryLedgerResource::collection($ledgers))->addAdditional('fnskuProductBalance', [
            'reconciled_quantity' => $fnskuProduct->reconciled_quantity,
        ]);
    }

    public function unreconcileAll(AmazonIntegrationInstance $integrationInstance): Response
    {
        (new AmazonLedgerManager($integrationInstance))->unreconcileLedgers();
        return $this->response->setMessage('Ledger unreconciliation process completed');
    }

    /**
     * @throws Exception
     */
    public function getLastReconciledDate(AmazonIntegrationInstance $integrationInstance): string
    {
        return (new AmazonLedgerManager($integrationInstance))->getLastReconciledDate()->toDateString();
    }

    /**
     * @throws Exception
     */
    public function getReconciliationReport(AmazonIntegrationInstance $integrationInstance)
    {
        return (new AmazonLedgerManager($integrationInstance))->reconciliationReport();
    }

    /**
     * @throws Throwable
     */
    public function createInventoryAdjustmentFromLedger(Request $request, AmazonIntegrationInstance $integrationInstance, AmazonFbaReportInventoryLedger $ledger): Response
    {
        $request->validate([
            'notes' => 'required|string',
        ]);
        $ledger = (new AmazonLedgerManager($integrationInstance))->createInventoryAdjustmentFromLedger($ledger, $request->get('notes'));
        return $this->response->addData(AmazonFbaReportInventoryLedgerResource::make($ledger));
    }
}
