<?php

namespace Modules\Amazon\Http\Controllers;

use App\Data\IdSelectionData;
use App\DataTable\DataTableResource;
use App\Response;
use Exception;
use Modules\Amazon\Entities\AmazonFbaInboundShipment;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Http\Controllers\Concerns\AmazonExportableControllerTrait;
use Modules\Amazon\Http\Resources\AmazonFbaInboundShipmentResource;
use Modules\Amazon\Managers\AmazonInboundManager;
use Modules\Amazon\Repositories\AmazonFbaInboundShipmentRepository;

class AmazonFbaInboundShipmentController extends AbstractAmazonDataTableController
{
    use AmazonExportableControllerTrait;

    private AmazonFbaInboundShipmentRepository $shipments;

    public function __construct()
    {
        $this->shipments = app(AmazonFbaInboundShipmentRepository::class);
        parent::__construct();
    }

    protected function getResource(): DataTableResource|string
    {
        return AmazonFbaInboundShipmentResource::class;
    }

    protected function getModel(): string
    {
        return AmazonFbaInboundShipment::class;
    }

    /**
     * @throws Exception
     */
    public function refresh(AmazonIntegrationInstance $integrationInstance): Response
    {
        (new AmazonInboundManager($integrationInstance))->refreshShipments();

        return $this->response->setMessage('Inbound shipments downloading...');
    }

    public function show(
        AmazonIntegrationInstance $integrationInstance,
        AmazonFbaInboundShipment $shipment
    ): ?Response {
        $shipment->load([
            'amazonFbaInboundShipmentItems.amazonProduct',
            'skuLink',
            'ledgers'
        ]);

        return $this->response->addData(AmazonFbaInboundShipmentResource::make($shipment));
    }

    public function getUnlinkedShipments(AmazonIntegrationInstance $integrationInstance)
    {
        $amazonFbaInboundShipments = $this->shipments->getUnlinkedShipments($integrationInstance);

        return $this->response->addData($amazonFbaInboundShipments);
    }

    /**
     * @throws Exception
     */
    public function process(IdSelectionData $data, AmazonIntegrationInstance $integrationInstance): Response
    {
        $processedShipments = (new AmazonInboundManager($integrationInstance))->process($data->ids);
        return $this->response->setMessage('Inbound shipments processed')->addData($this->getResource()::collection($processedShipments));
    }

    /**
     * @throws Exception
     */
    public function processAll(AmazonIntegrationInstance $integrationInstance): Response
    {
        (new AmazonInboundManager($integrationInstance))->processUnprocessed();
        return $this->response->setMessage('Inbound shipments processed');
    }

    /**
     * @throws Exception
     */
    public function unprocess(IdSelectionData $data, AmazonIntegrationInstance $integrationInstance): Response
    {
        (new AmazonInboundManager($integrationInstance))->unprocess($data->ids);
        return $this->response->setMessage('Inbound shipments unprocessed');
    }

    public function setSentBeforeInitialCount(IdSelectionData $data): Response
    {
        $this->shipments->setValueForIds($data->ids, AmazonFbaInboundShipment::class, [
            'is_before_initial_count' => $data->value == 'true',
            'errorLog' => null,
        ]);

        return $this->response->setMessage('Inbound shipments updated');
    }
}
