<?php

namespace Modules\Amazon\Entities;

use App\Abstractions\UniqueFieldsInterface;
use App\Models\User;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\MassAssignmentException;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Modules\Amazon\Enums\Entities\ReportProcessingStatusEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;

/**
 * Class AmazonReport.
 *
 * @property int $id
 * @property int $integration_instance_id
 * @property string $reportId
 * @property AmazonReportTypeEnum $reportType
 * @property Carbon $dataStartTime
 * @property Carbon $dataEndTime
 * @property array $options
 * @property string $reportScheduledId
 * @property Carbon $createdTime
 * @property string $processingStatus
 * @property Carbon $processingStartTime
 * @property Carbon $processingEndTime
 * @property array $marketplaceIds
 * @property string $reportDocumentId
 * @property string $filename
 * @property Carbon $processed_at
 * @property int $requested_by_user_id
 * @property-read AmazonIntegrationInstance $integrationInstance
 * @property-read User $requestedByUser
 * @property Carbon $created_at
 * @property Carbon $updated_at
 *
 * @method static amazonReports()
 *
 * @mixin IdeHelperAmazonReport
 */
class AmazonReport extends Model implements UniqueFieldsInterface
{
    use HasFactory;

    const BULK_THRESHOLD = 1000;

    protected $casts = [
        'marketplaceIds' => 'array',
        'integration_instance_id' => 'integer',
        'reportId' => 'string',
        'reportType' => AmazonReportTypeEnum::class,
        'dataStartTime' => 'datetime',
        'dataEndTime' => 'datetime',
        'options' => 'array',
        'reportScheduledId' => 'string',
        'createdTime' => 'datetime',
        'processingStatus' => ReportProcessingStatusEnum::class,
        'processingStartTime' => 'datetime',
        'processingEndTime' => 'datetime',
        'reportDocumentId' => 'string',
        'processed_at' => 'datetime',
    ];

    protected $guarded = [];

    public static array $adjustmentReasons = [
        ['Code' => '6', 'Type' => '-', 'Reason' => 'Damaged at Amazon fulfillment center', 'Definition' => 'A decrease to your carrier-damaged inventory level. This code is always followed by a P code increase to your fulfillment-center-damaged inventory level.'],
        ['Code' => '7', 'Type' => '-', 'Reason' => 'Damaged at Amazon fulfillment center', 'Definition' => 'A decrease to your expired inventory level. This code is always followed by a P code increase to your fulfillment-center-damaged inventory level.'],
        ['Code' => 'E', 'Type' => '-', 'Reason' => 'Damaged at Amazon fulfillment center', 'Definition' => 'A decrease to your sellable inventory level. This code is always followed by a P code increase to your fulfillment-center-damaged inventory level.'],
        ['Code' => 'H', 'Type' => '-', 'Reason' => 'Damaged at Amazon fulfillment center', 'Definition' => 'A decrease to your customer-damaged inventory level. This code is always followed by a P code increase to your fulfillment-center-damaged inventory level.'],
        ['Code' => 'K', 'Type' => '-', 'Reason' => 'Damaged at Amazon fulfillment center', 'Definition' => 'A decrease to your defective inventory level. This code is always followed by a P code increase to your fulfillment-center-damaged inventory level.'],
        ['Code' => 'U', 'Type' => '-', 'Reason' => 'Damaged at Amazon fulfillment center', 'Definition' => 'A decrease to your distributor-damaged inventory level. This code is always followed by a P code increase to your fulfillment-center-damaged inventory level.'],
        ['Code' => 'D', 'Type' => '-', 'Reason' => 'Inventory disposed of', 'Definition' => 'A decrease to your inventory level because inventory has been disposed of'],
        ['Code' => 'F', 'Type' => '+', 'Reason' => 'Inventory found', 'Definition' => 'An increase to your inventory level because missing inventory has been found'],
        ['Code' => 'N', 'Type' => '+', 'Reason' => 'Inventory found', 'Definition' => 'An increase to your inventory level because inventory that was incorrectly assigned has been transferred to your account or because you received an Amazon reimbursement'],
        ['Code' => 'M', 'Type' => '-', 'Reason' => 'Inventory misplaced', 'Definition' => 'A decrease to your inventory level because inventory is missing from a bin location in a fulfillment center'],
        ['Code' => '5', 'Type' => '-', 'Reason' => 'Inventory misplaced', 'Definition' => 'A decrease to your inventory level because inventory is missing from a bin location in a fulfillment center'],
        ['Code' => '3', 'Type' => '+', 'Reason' => 'Product redefinition transfer in', 'Definition' => 'Two products with separate identifiers (SKUs) are determined to be the same item. One SKU will be removed from your inventory (code 4) and added (code 3) as the other SKU.'],
        ['Code' => '4', 'Type' => '-', 'Reason' => 'Product redefinition transfer out', 'Definition' => 'Two products with separate identifiers (SKUs) are determined to be the same item. One SKU will be removed from your inventory (code 4) and added (code 3) as the other SKU.'],
        ['Code' => 'O', 'Type' => '-', 'Reason' => 'Inventory correction', 'Definition' => 'Two products with separate identifiers (SKUs) are determined to be the same item. One SKU will be removed from your inventory (code 4) and added (code 3) as the other SKU.'],
        ['Code' => 'P', 'Type' => '+', 'Reason' => 'Inventory disposition change', 'Definition' => 'Units of a certain disposition are added to your inventory after the removal from a different disposition. This code always follows a code 6, 7, E, H, K, U, or Q adjustment.'],
        ['Code' => 'Q', 'Type' => '-', 'Reason' => 'Inventory disposition change', 'Definition' => 'Units of a certain disposition are removed from your inventory and then added back to your inventory as a different disposition. This code is always followed by a P code adjustment.'],
    ];

    public static function getUniqueFields(): array
    {
        return ['integration_instance_id', 'reportId'];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function integrationInstance(): BelongsTo
    {
        return $this->belongsTo(AmazonIntegrationInstance::class, 'integration_instance_id');
    }

    public function requestedByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'requested_by_user_id');
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    /**
     * Add check for report type to avoid syncing unexpected reports.
     */
    public function scopeAmazonReports(Builder $builder): Builder
    {
        $builder->whereIn('reportType', AmazonReportTypeEnum::values());

        return $builder;
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    /**
     * @throws MassAssignmentException
     */
    public function updateFilename(string $filename): bool
    {
        return $this->update(['filename' => $filename]);
    }
}
