<?php

namespace Modules\Amazon\Entities;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelProduct;
use App\Abstractions\Integrations\SalesChannels\SalesChannelProductManagerInterface;
use App\Abstractions\Integrations\SalesChannels\SalesChannelProductRepositoryInterface;
use App\DataTable\Exports\DataTableExporter as Exportable;
use App\DTO\ProductDto;
use App\DTO\ProductImageDto;
use App\Importers\Parsers\FieldParserFactory;
use App\Integrations\Listings\ListingsHelper;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Product;
use App\Models\ProductListing;
use App\Repositories\DataImportMappingRepository;
use Awobaz\Compoships\Compoships;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Arr;
use Kirschbaum\PowerJoins\PowerJoins;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetReportsAdt;
use Modules\Amazon\Database\Factories\AmazonProductFactory;
use Modules\Amazon\Enums\Entities\AmazonProductFulfillmentChannelEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Http\Resources\AmazonProductResource;
use Modules\Amazon\Managers\AmazonProductManager;
use Modules\Amazon\Repositories\AmazonProductRepository;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Spatie\LaravelData\DataCollection;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property string $item_name
 * @property string $seller_sku
 * @property float $price
 * @property int $quantity
 * @property int $removed_from_amazon
 * @property string $image_url
 * @property int $product_id_Type
 * @property string $item_note
 * @property int $item_condition
 * @property string $asin1
 * @property int $product_id
 * @property string $fulfillment_channel
 * @property string $merchant_shipping_group
 * @property string $status
 * @property array $json_object
 * @property array $catalog_data
 * @property ?Carbon $catalog_data_last_sync
 * @property bool $was_catalog_data_sync_attempted
 * @property string $brand
 * @property string $manufacturer
 * @property string $model
 * @property string $part_number
 * @property string $product_type
 * @property string $sales_rank
 * @property string $color
 * @property string $package_length
 * @property string $package_width
 * @property string $package_height
 * @property string $package_dimensions_unit
 * @property string $package_weight
 * @property string $package_weight_unit
 * @property string $main_image
 * @property string $browse_classification
 * @property string $website_group_name
 * @property string $style
 * @property string $size
 * @property string $item_length
 * @property string $item_width
 * @property string $item_height
 * @property string $item_dimensions_unit
 * @property string $item_weight
 * @property string $item_weight_unit
 * @property int $number_of_items
 * @property string $department
 * @property string $item_type_name
 * @property string $material
 * @property string $item_type_keyword
 * @property int $packageQuantity
 * @property string $item_description
 * @property string $listing_id
 * @property string $open_date
 * @property Carbon $created_at
 * @property Carbon|null $updated_at
 * @property-read ProductListing $productListing
 * @property-read AmazonIntegrationInstance $integrationInstance
 * @property-read AmazonFbaReportInventory $fbaInventory
 *
 * @mixin IdeHelperAmazonProduct
 */
class AmazonProduct extends AbstractSalesChannelProduct
{
    use BulkImport;
    use Compoships;
    use HasFactory;
    use HasFilters;
    use HasSort;
    use ListingsHelper;
    use PowerJoins;

    protected $guarded = [];

    protected $casts = [
        'json_object' => 'array',
        'catalog_data' => 'array',
        'catalog_data_last_sync' => 'datetime',
        'was_catalog_data_sync_attempted' => 'bool',
        'fulfillment_channel' => AmazonProductFulfillmentChannelEnum::class,
    ];

    const ALLOWED_SKU_PRODUCT_TYPES = [
        Product::TYPE_STANDARD,
        Product::TYPE_KIT,
        Product::TYPE_BLEMISHED,
    ];

    public string $dataTableClass = self::class;

    public static function getIntegrationName(): string
    {
        return 'amazon';
    }

    public static function getLastModified(): ?string
    {
        return null;
    }

    public static function getUniqueField(): string
    {
        return 'seller_sku';
    }

    public function isUsed(): bool|array
    {
        // If a listing has amazon orders, we don't allow the deletion (SKU-2680)
        $linkedOrders = AmazonOrder::with([])
            ->whereHas('orderItems', function ($query) {
                $query->where('SellerSKU', $this->seller_sku);
            })
            ->count();

        if ($linkedOrders) {
            return [
                'amazon.orders' => trans_choice('messages.currently_used', $linkedOrders, [
                    'resource' => 'Amazon Order',
                    'model' => 'Amazon Listing('.$this->seller_sku.')',
                ]),
            ];
        }

        return false;
    }

    public function isActive(): bool
    {
        return $this->status === 'Active';
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function integrationInstance(): BelongsTo
    {
        return $this->belongsTo(AmazonIntegrationInstance::class);
    }

    public function amazonIntegrationInstance(): BelongsTo
    {
        return $this->integrationInstance();
    }

    public function fbaInventory(): HasOne
    {
        return $this->hasOne(AmazonFbaReportInventory::class, 'sku', 'seller_sku');
    }

    /*
    |--------------------------------------------------------------------------
    | Implementers
    |--------------------------------------------------------------------------
    */

    public function availableColumns(): array
    {
        return [
            "id" => "bigint",
            "product" => "string", // for product link
            "item_name" => "varchar",
            "seller_sku" => "varchar",
            "price" => "decimal",
            "quantity" => "varchar",
            "image_url" => "varchar",
            "product_id_type" => "varchar",
            "item_note" => "varchar",
            "item_condition" => "varchar",
            "asin1" => "varchar",
            "product_id" => "varchar",
            "fulfillment_channel" => "varchar",
            "merchant_shipping_group" => "varchar",
            "status" => "varchar",
            "json_object" => "longtext",
            "catalog_data" => "longtext",
            "brand" => "varchar",
            "manufacturer" => "varchar",
            "model" => "varchar",
            "part_number" => "varchar",
            "product_type" => "varchar",
            "sales_rank" => "varchar",
            "color" => "varchar",
            "package_length" => "varchar",
            "package_width" => "varchar",
            "package_height" => "varchar",
            "package_dimensions_unit" => "varchar",
            "package_weight" => "varchar",
            "package_weight_unit" => "varchar",
            "main_image" => "varchar",
            "browse_classification" => "varchar",
            "open_date" => "varchar",
            "style" => "varchar",
            "size" => "varchar",
            "item_length" => "varchar",
            "item_width" => "varchar",
            "item_height" => "varchar",
            "item_dimensions_unit" => "varchar",
            "item_weight" => "varchar",
            "item_weight_unit" => "varchar",
            "number_of_items" => "varchar",
            "department" => "varchar",
            "item_type_name" => "varchar",
            "item_type_keyword" => "varchar",
            "packageQuantity" => "varchar",
            "listing_id" => "varchar",
            "item_description" => "text",
            "material" => "varchar",
            "website_group_name" => "varchar",
            "catalog_data_last_sync" => "datetime",
            "was_catalog_data_sync_attempted" => "tinyint",
            "removed_from_amazon" => "tinyint",
            "mapped_sku" => "varchar", // same as product
            "created_at" => "timestamp",
            "updated_at" => "timestamp",
            "mapped_at" => "timestamp", // when the product listing was created
        ];
    }

    public function filterableColumns(): array
    {
        $filterableColumns = array_keys($this->availableColumns());
        unset($filterableColumns[array_search('mapped_sku', $filterableColumns)]);
        return $filterableColumns;
    }

    public function visibleColumns(): array
    {
        return [
            "seller_sku",
            "status",
            "item_name",
            "asin1",
            "product",
            "price",
            "quantity",
            "brand",
            "product_type",
            "sales_rank",
            "style",
            "created_at"
        ];
    }

    public function generalFilterableColumns(): array
    {
        return ['id', 'seller_sku', 'asin1'];
    }

    /*
    |--------------------------------------------------------------------------
    | DataTables
    |--------------------------------------------------------------------------
    */

    public static function getResource(): string
    {
        return AmazonProductResource::class;
    }

    /*
    |--------------------------------------------------------------------------
    | Other functions
    |--------------------------------------------------------------------------
    */

    public function getListingUrl(): ?string
    {
        return "https://www.amazon.com/dp/{$this->asin1}?psc=1";
    }

    public static function getExportableFields(): array
    {
        return [
            'seller_sku' => Exportable::makeExportableField('seller_sku', false),
            'item_name' => Exportable::makeExportableField('item_name', false),
            'sales_rank' => Exportable::makeExportableField('sales_rank', false),
            'product_type' => Exportable::makeExportableField('product_type', false),
            'item_description' => Exportable::makeExportableField('item_description', false),
            'modelNumber' => Exportable::makeExportableField('modelNumber', false),
            'material' => Exportable::makeExportableField('material', false),
            'listing_id' => Exportable::makeExportableField('listing_id', false),
            'manufacturer' => Exportable::makeExportableField('manufacturer', false),
            'brand' => Exportable::makeExportableField('brand', false),
            'price' => Exportable::makeExportableField('price', false),
            'product.id' => Exportable::makeExportableField('product.id', false),
            'product.name' => Exportable::makeExportableField('product.name', false),
            'product.sku' => Exportable::makeExportableField('product.sku', false),
            'open_date' => Exportable::makeExportableField('open-date', false),
            'asin1' => Exportable::makeExportableField('asin1', false),
            'quantity' => Exportable::makeExportableField('quantity', false),
            'merchant_shipping_group' => Exportable::makeExportableField('merchant_shipping_group', false),
            'item_condition' => Exportable::makeExportableField('item_condition', false),
            'fulfillment_channel' => Exportable::makeExportableField('fulfillment_channel', false),
            'mapped_sku' => Exportable::makeExportableField('mapped_sku', false),
            'created_at' => Exportable::makeExportableField('created_at', false),
            'updated_at' => Exportable::makeExportableField('updated_at', false),
            'mapped_at' => Exportable::makeExportableField('mapped_at', false),
        ];
    }

    public function scopeFilterDuplicateAsins(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $duplicateAsins = self::query()->select('asin1')
            ->groupBy('asin1')
            ->havingRaw('COUNT(*) > 1')
            ->pluck('asin1');
        /*
         * return only products that have duplicate asins
         */
        if ($value) {
            return $builder->whereIn('asin1', $duplicateAsins);
        } else {
            return $builder->whereNotIn('asin1', $duplicateAsins);
        }
    }

    public function scopeFilterAsinMultipleProducts(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $multipleProducts = self::query()
            ->select('asin1')
            ->whereHas('productListing')
            ->groupBy('asin1')
            ->havingRaw('COUNT(*) > 1')
            ->pluck('asin1');
        /*
         * return only products that have duplicate asins
         */
        if ($value) {
            return $builder->whereIn('asin1', $multipleProducts);
        } else {
            return $builder->whereNotIn('asin1', $multipleProducts);
        }
    }

    public function delete(): ?bool
    {
        $this->productListing()->delete();

        return parent::delete();
    }

    /**
     * @throws NotFoundExceptionInterface
     * @throws ContainerExceptionInterface
     */
    public function getProductDto(): ProductDto
    {
        return ProductDto::from([
            'sku' => $this->applyDataMapping($this, 'sku'),
            'name' => $this->applyDataMapping($this, 'item_name'),
            'barcode' => $this->applyDataMapping($this, 'barcode'),
            'weight' => $this->applyDataMapping($this, 'weight') ?? 0,
            'images' => $this->getImages(),
            'pricing' => $this->getPricing(),
        ]);
    }

    public function getImages(): DataCollection
    {
        $images = data_get($this->catalog_data, 'catalog_data.images.0.images', []);
        return ProductImageDto::collection(Arr::map($images, function ($image) {
            return ProductImageDto::from([
                'url' => $image['link'],
                'name' => $image['variant'],
                'sku' => $this->applyDataMapping($this, 'sku'),
            ]);
        }));
    }

    /**
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function applyDataMapping(mixed $data, string $mappedColumn): mixed
    {
        if (!$mappings = config()->get('amazon_mappings')) {
            $mappings = app(DataImportMappingRepository::class)->getListingMappings($this->integration_instance_id);
        }

        $skuMapping = $mappings->where('sku_field', $mappedColumn)->first();
        $listingField = @$skuMapping['listing_field'];

        /*
         * TODO: I believe these are default mappings but the code doesn't make it obvious.  I would get default mappings
         *  in a separate method so that the method name makes it clear what is being done
         */

        // One idea for accessing defaults
        /*if (is_null($listingField))
        {
            $mappings =
            $listingField = $mappings[$listingField];
        }*/
        if (is_null($listingField)) {
            switch ($mappedColumn) {
                case 'item_name':
                case 'title':
                    $listingField = 'item_name';
                    break;

//                case 'sales_channel_listing_id':
//                    $listingField = 'seller_sku';
//                    break;

                case 'sku':
                    $listingField = 'seller_sku';
                    break;

                case 'price':
                    $listingField = 'price';
                    break;

                case 'brand_name':
                    $listingField = 'brand';
                    break;

                case 'weight':
                    $listingField = 'package_weight';
                    break;

                case 'weight_unit':
                    $listingField = 'package_weight_unit';
                    break;

                case 'length':
                    $listingField = 'package_length';
                    break;

                case 'width':
                    $listingField = 'package_width';
                    break;

                case 'height':
                    $listingField = 'package_height';
                    break;

                case 'dimension_unit':
                    $listingField = 'package_dimensions_unit';
                    break;
            }
        }

        //Return value
        $returnValue = @$data[$listingField];

        //Apply mappings
        if ($returnValue) {
            if (@$skuMapping['parsers'] && is_array($skuMapping['parsers'])) {
                foreach ($skuMapping['parsers'] as $parser) {
                    $returnValue = FieldParserFactory::make(
                        $parser['rule'],
                        $parser['args'],
                        $data
                    )->parse($returnValue);
                }
            }
        }

        return $returnValue;
    }

    protected static function newFactory()
    {
        return AmazonProductFactory::new();
    }

    public static function getSkuField(): string
    {
        return 'seller_sku';
    }

    public static function getPriceField(): string
    {
        return 'price';
    }

    public function getIsFbaAttribute()
    {
        return $this->fulfillment_channel?->value === 'AMAZON_NA';
    }

    /**
     * @throws Exception
     */
    public function manager(IntegrationInstanceInterface $integrationInstance): SalesChannelProductManagerInterface
    {
        return new AmazonProductManager($integrationInstance);
    }

    public function refreshAdt(): ApiDataTransformerInterface
    {
        return new AmazonGetReportsAdt(AmazonReportTypeEnum::PRODUCTS);
    }

    public static function repository(): SalesChannelProductRepositoryInterface
    {
        return app(AmazonProductRepository::class);
    }
}
