<?php

namespace Modules\Amazon\Entities;

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelOrderLine;
use App\Data\SalesOrderLineData;
use App\Helpers;
use App\Models\FinancialLine;
use App\Models\Setting;
use App\Repositories\WarehouseRepository;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphMany;

/**
 * Class AmazonOrderItem.
 *
 * @property int $id
 * @property int $amazon_order_id
 * @property array $json_object
 * @property-read string $ASIN
 * @property-read string $SellerSKU
 * @property-read string $OrderItemId
 * @property-read string $Title
 * @property-read int $QuantityOrdered
 * @property-read int $QuantityShipped
 * @property-read float $ItemCurrency
 * @property-read float $ItemPrice
 * @property-read float $ShippingPrice
 * @property-read float $ItemTax
 * @property-read float $ShippingTax
 * @property-read float $ShippingDiscount
 * @property-read float $ShippingDiscountTax
 * @property-read float $PromotionDiscount
 * @property-read float $PromotionDiscountTax
 * @property-read float $CODFee
 * @property-read float $CODFeeDiscount
 * @property-read string $IsGift
 * @property-read string $ConditionNote
 * @property-read string $ConditionId
 * @property-read string $ConditionSubtypeId
 * @property-read string $ScheduledDeliveryStartDate
 * @property-read string $ScheduledDeliveryEndDate
 * @property-read string $PriceDesignation
 * @property-read string $SerialNumberRequired
 * @property-read string $IsTransparency
 * @property-read string $IossNumber
 * @property-read string $StoreChainStoreId
 * @property-read string $DeemedResellerCategory
 * @property-read AmazonOrder $order
 * @property-read AmazonProduct $product
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 */
class AmazonOrderItem extends AbstractSalesChannelOrderLine
{
    use HasFactory;

    protected WarehouseRepository $warehouseRepository;

    public function __construct(array $attributes = [])
    {
        $this->warehouseRepository = app(WarehouseRepository::class);
        parent::__construct($attributes);
    }

    public static function getUniqueId(): string
    {
        return 'OrderItemId';
    }

    public static function getTableUniqueId(): string
    {
        return 'OrderItemId';
    }

    public static function getLineItemsKey(): ?string
    {
        // Not implemented because the call to get items is separate
        return null;
    }

    public static function getParentRelationId(): string
    {
        return 'amazon_order_id';
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function order(): BelongsTo
    {
        return $this->belongsTo(AmazonOrder::class, 'amazon_order_id');
    }

    public function amazonOrder(): BelongsTo
    {
        return $this->order();
    }

    public function financialLines(): MorphMany
    {
        return $this->morphMany(FinancialLine::class, 'external_source');
    }

    public function product(): \Awobaz\Compoships\Database\Eloquent\Relations\BelongsTo
    {
        // TODO: Verify this works
        return $this->belongsTo(
            AmazonProduct::class,
            ['integration_instance_id', 'SellerSKU'],
            ['integration_instance_id', 'seller_sku']
        );
    }

    public function getAmount(): float
    {
        return $this->QuantityOrdered > 0 ? ($this->ItemPrice / $this->QuantityOrdered) : 0;
    }

    /**
     * We do not try to calculate a tax rate, instead we rely on the tax amount (aka tax allocation in sku).
     * Tax allocation is the sum of taxes for the entire line.  No need to divide by quantity ordered.
     */
    public function getTaxAllocation(): float
    {
        return $this->ItemTax - $this->PromotionDiscountTax;
    }

    /**
     * Discount allocation is the sum of discounts for the entire line.  No need to divide by quantity ordered.
     */
    public function getDiscountAllocation(): float
    {
        return $orderItem->PromotionDiscount ?? 0;
    }

    public function getWarehouseId(): ?int
    {
        // TODO: This is a temporary patch, please remove
        /** @var AmazonProduct $amazonProduct */
        $amazonProduct = AmazonProduct::query()
            ->where('seller_sku', $this->SellerSKU)
            ->where('integration_instance_id', $this->order->integrationInstance->id)
            ->first();

        $product = $amazonProduct?->productListing?->product;

        if (! $product) {
            return null;
        }

        if ($this->order->FulfillmentChannel === 'AFN') {
            return $this->order->integrationInstance->warehouse->id;
        }

        return (new WarehouseRepository())->getPriorityWarehouseIdForProduct(
            $product,
            $this->QuantityOrdered,
        );
    }

    public function getNominalCodeId(): ?int
    {
        return $this->amazonOrder->integrationInstance->getSalesNominalCodeId() ?? Helpers::setting(Setting::KEY_NC_MAPPING_SALES_ORDERS);
    }

    public function getSalesOrderLineData(): SalesOrderLineData
    {
        $order = $this->order;

        return SalesOrderLineData::from([
            'sales_order_number' => $order->AmazonOrderId,
            'sales_channel_line_id' => $this->{self::getTableUniqueId()},
            'amount' => $this->getAmount(),
            'tax_allocation' => $this->getTaxAllocation(),
            'discount_allocation' => $this->getDiscountAllocation(),
            'description' => $this->Title,
            'product_id' => $this->productListingsProductId,
            'product_listing_id' => $this->productListingsId,
            'is_product' => true,
            'quantity' => $this->QuantityOrdered,
            'warehouse_id' => $this->getWarehouseId(),
            'nominal_code_id' => $this->getNominalCodeId(),
        ]);
    }

    public function salesChannelProductClass(): string
    {
        return AmazonProduct::class;
    }

    public static function getSalesChannelProductUniqueId(): string
    {
        return 'seller_sku';
    }
}
