<?php

namespace Modules\Amazon\Entities;

use App\Abstractions\UniqueFieldsInterface;
use Exception;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Modules\Amazon\Database\Factories\AmazonNewFbaInboundShipmentItemFactory;
use Modules\Amazon\Enums\Entities\AmazonFbaInboundShipmentItemLabelOwnerEnum;
use Modules\Amazon\Enums\Entities\AmazonProductFulfillmentChannelEnum;

/**
 * @property int $id
 * @property int $amazon_new_fba_inbound_shipment_id
 * @property string $msku
 * @property string $fnsku
 * @property int $quantity
 * @property string $asin
 * @property AmazonFbaInboundShipmentItemLabelOwnerEnum $labelOwner
 * @property array $prepInstructions
 * @property array $json_object
 *
 * @property-read AmazonNewFbaInboundShipment $amazonFbaInboundShipment
 * @property-read AmazonProduct $amazonProduct
 *
 */
class AmazonNewFbaInboundShipmentItem extends Model implements UniqueFieldsInterface
{
    use HasFactory;

    protected $guarded = [];

    protected $casts = [
        'labelOwner' => AmazonFbaInboundShipmentItemLabelOwnerEnum::class,
        'prepInstructions' => 'array',
        'json_object' => 'array',
    ];

    /*
    |--------------------------------------------------------------------------
    | Implementers
    |--------------------------------------------------------------------------
    */
    public static function getUniqueFields(): array
    {
        return [
            'amazon_new_fba_inbound_shipment_id',
            'msku',
        ];
    }

    public static function newFactory(): Factory
    {
        return AmazonNewFbaInboundShipmentItemFactory::new();
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function amazonFbaInboundShipment(): BelongsTo
    {
        return $this->belongsTo(AmazonNewFbaInboundShipment::class, 'amazon_new_fba_inbound_shipment_id', 'id');
    }

    /**
     * @throws Exception
     */
    public function amazonProduct(): BelongsTo
    {
        return $this->belongsTo(AmazonProduct::class, 'msku', 'seller_sku')
            ->selectRaw('amazon_products.*')
            ->join('amazon_new_fba_inbound_shipments', 'amazon_products.integration_instance_id', '=', 'amazon_new_fba_inbound_shipments.integration_instance_id')
            ->whereColumn(
                'amazon_new_fba_inbound_shipments.integration_instance_id', 'amazon_products.integration_instance_id'
            )
            ->where('amazon_products.fulfillment_channel', '!=', AmazonProductFulfillmentChannelEnum::DEFAULT);
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public function getFnskuProduct(): ?AmazonFnskuProduct
    {
        return AmazonFnskuProduct::with('product.productInventory')
            ->where('integration_instance_id', $this->amazonFbaInboundShipment->integration_instance_id)
            ->where('fnsku', $this->fnsku)
            ->where('disposition', 'SELLABLE')
            ->where('location', 'US')
            ->first();
    }
}
