<?php

namespace Modules\Amazon\Entities;

use App\Abstractions\Integrations\IntegrationModelInterface;
use App\DataTable\DataTableModelInterface;
use App\DataTable\DataTableModelTrait;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Awobaz\Compoships\Compoships;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Modules\Amazon\Abstractions\AmazonInboundShipmentSourceInterface;
use Modules\Amazon\Database\Factories\AmazonNewFbaInboundShipmentFactory;
use Modules\Amazon\Enums\Entities\AmazonInboundDestinationTypeEnum;
use Modules\Amazon\Enums\Entities\AmazonNewFbaInboundShipmentStatusEnum;

/**
 * @property int $id
 * @property int $amazon_new_fba_inbound_plan_id
 * @property int $integration_instance_id
 * @property ?int $sku_link_id
 * @property ?string $sku_link_type
 * @property string $name
 * @property string $sourceName
 * @property string $shipmentId
 * @property AmazonNewFbaInboundShipmentStatusEnum $status
 * @property string $amazonReferenceId
 * @property string $shipmentConfirmationId
 * @property string $destinationCountry
 * @property string $destinationWarehouse
 * @property AmazonInboundDestinationTypeEnum $destinationType
 * @property array $destination
 * @property array $source
 * @property string $selectedTransportationOptionId
 * @property string $placementOptionId
 * @property ?string $errorLog
 * @property bool $is_before_initial_count // Needed due to https://github.com/amzn/selling-partner-api-docs/issues/2706
 * @property array $json_object
 * @property Carbon $created_at
 * @property ?Carbon $updated_at
 *
 * @property-read AmazonNewFbaInboundPlan $amazonFbaInboundPlan
 * @property-read Collection|AmazonNewFbaInboundShipmentItem[] $amazonFbaInboundShipmentItems
 * @property-read AmazonInboundShipmentSourceInterface $skuLink
 * @property-read Collection|AmazonFbaReportInventoryLedger[] $ledgers
 * @property-read AmazonFbaInboundShipFromMapping $shipFromMapping
 *
 */
class AmazonNewFbaInboundShipment extends Model implements Filterable, IntegrationModelInterface, DataTableModelInterface, Sortable
{
    use Compoships;
    use DataTableModelTrait;
    use HasFactory;
    use HasFilters;
    use HasSort;

    protected $guarded = [];

    protected $casts = [
        'status' => AmazonNewFbaInboundShipmentStatusEnum::class,
        'destinationType' => AmazonInboundDestinationTypeEnum::class,
        'json_object' => 'array',
        'destination' => 'array',
        'source' => 'array',
    ];

    /*
    |--------------------------------------------------------------------------
    | Implementers
    |--------------------------------------------------------------------------
    */

    public static function getUniqueField(): string
    {
        return 'shipmentId';
    }

    public static function newFactory(): Factory
    {
        return AmazonNewFbaInboundShipmentFactory::new();

    }

    public static function specialLabels(): array
    {
        return [
            'id' => 'ID',
            'asin' => 'ASIN',
            'destinationWarehouse' => 'FCID',
            'is_before_initial_count' => 'Sent Before Initial Count?',
            'sourceName' => 'Source',
            'destinationCountry' => 'Country',
            'sku_link' => 'SKU Link',
            'sku_link_exists' => 'SKU Link Exists?',
        ];
    }

    public function availableColumns(): array
    {
        return [
            'id' => 'int',
            'shipmentConfirmationId' => 'string',
            'sku_link' => 'sku_link',
            'sku_link_exists' => 'checkbox',
            'sourceName' => 'string',
            'status' => 'string',
            'shipmentId' => 'string',
            'is_before_initial_count' => 'checkbox',
            'errorLog' => 'string',
            'name' => 'string',
            'destinationCountry' => 'string',
            'destinationWarehouse' => 'string',
            'destinationType' => 'string',
            'selectedTransportationOptionId' => 'string',
            'placementOptionId' => 'string',
            'created_at' => 'datetime',
            'updated_at' => 'datetime',
        ];
    }

    public function scopeColumns(): array
    {
        return [
            'sku_link_exists'
        ];
    }

    public function filterableColumns(): array
    {
        return array_merge(array_filter($this->visibleColumns(), fn($value) => $value !== 'sku_link'), ['sku_link_exists']);
    }

    public function visibleColumns(): array
    {
        return [
            'id',
            'shipmentConfirmationId',
            'sku_link',
            'status',
            'sourceName',
            'name',
            'destinationWarehouse',
            'destinationCountry',
            'destinationType',
            'is_before_initial_count',
            'errorLog',
            'created_at',
            'updated_at',
        ];
    }

    public function generalFilterableColumns(): array
    {
        return ['shipmentConfirmationId', 'sourceName'];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function integrationInstance(): BelongsTo
    {
        return $this->belongsTo(AmazonIntegrationInstance::class);
    }

    public function amazonFbaInboundPlan(): BelongsTo
    {
        return $this->belongsTo(AmazonNewFbaInboundPlan::class, 'amazon_new_fba_inbound_plan_id', 'id');
    }

    public function amazonFbaInboundShipmentItems(): HasMany
    {
        return $this->hasMany(AmazonNewFbaInboundShipmentItem::class);
    }

    public function skuLink(): MorphTo
    {
        return $this->morphTo('sku_link');
    }

    public function ledgers(): \Awobaz\Compoships\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(
            AmazonFbaReportInventoryLedger::class,
            ['integration_instance_id', 'reference_id'],
            ['integration_instance_id', 'shipmentId']
        );
    }

    public function shipFromMapping(): \Awobaz\Compoships\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(AmazonFbaInboundShipFromMapping::class,
            ['integration_instance_id', 'sourceName'],
            ['integration_instance_id', 'name']
        );
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeFilterSkuLinkExists(Builder $builder, $operator, $value, $conjunction): Builder
    {
        if ($value) {
            $builder->whereNotNull('sku_link_id');
        } else {
            $builder->whereNull('sku_link_id');
        }

        return $builder;
    }
}
