<?php

namespace Modules\Amazon\Entities;

use App\Abstractions\Integrations\SalesChannels\SalesChannelFinancialEventInterface;
use App\Models\ExternalSourceFinancialLine;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Carbon;
use Modules\Amazon\Database\Factories\AmazonOrderFactory;
use Modules\Amazon\Enums\Entities\AmazonFinancialEventTypeEnum;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property int $amazon_financial_event_group_id
 * @property AmazonFinancialEventTypeEnum $event_type
 * @property bool $is_processed
 * @property array $json_object
 * @property-read AmazonIntegrationInstance $integrationInstance
 * @property-read AmazonFinancialEventGroup $amazonFinancialEventGroup
 * @property Carbon $created_at
 * @property ?Carbon $updated_at
 *
 * @mixin IdeHelperAmazonFinancialEvent
 */
abstract class AmazonFinancialEvent extends Model implements SalesChannelFinancialEventInterface
{
    protected $guarded = [];

    protected $casts = [
        'json_object' => 'array',
        'event_type' => AmazonFinancialEventTypeEnum::class,
    ];

    const BULK_THRESHOLD = 1000;

    public static function getIntegrationName(): string
    {
        return 'amazon';
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function integrationInstance(): BelongsTo
    {
        return $this->belongsTo(AmazonIntegrationInstance::class);
    }

    public function amazonFinancialEventGroup(): BelongsTo
    {
        return $this->belongsTo(AmazonFinancialEventGroup::class);
    }

    public function financialLines(): MorphMany
    {
        return $this->morphMany(ExternalSourceFinancialLine::class, 'external_source');
    }

    public function amazonOrder(): BelongsTo
    {
        return $this->belongsTo(AmazonOrder::class, 'AmazonOrderId', 'AmazonOrderId');
    }

    /**
     * @return Factory<static>
     */
    public static function newFactory(): Factory
    {
        return AmazonOrderFactory::new();
    }

    protected function aggregateFinancialLines(array $itemList, string $itemIdKey, string $orderItemId, string $feeListKey): array
    {
        $financialLines = [];

        collect($itemList)
            ->where($itemIdKey, $orderItemId)
            ->values()
            ->each(function ($event) use (&$financialLines, $feeListKey) {
                collect($event[$feeListKey])->each(function (array $chargeArray) use (&$financialLines) {
                    $chargeAmount = @$chargeArray['FeeAmount']['CurrencyAmount'];
                    $chargeType = @$chargeArray['FeeType'];

                    if ($chargeAmount != 0) {
                        if (isset($financialLines[$chargeArray['FeeType']])) {
                            $financialLines[$chargeType] = $financialLines[$chargeType] + $chargeAmount;
                        } else {
                            $financialLines[$chargeArray['FeeType']] = $chargeArray['FeeAmount']['CurrencyAmount'];
                        }
                    }
                });
            });

        return $financialLines;
    }
}
