<?php

namespace Modules\Amazon\Entities;

use App\DataTable\DataTableModelInterface;
use App\DataTable\DataTableModelTrait;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Models\InventoryAdjustment;
use App\Services\InventoryManagement\InventoryEvent;
use Awobaz\Compoships\Compoships;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Support\Collection;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property int $amazon_report_id
 * @property ?Carbon $reconciled_at
 * @property string $errorLog
 * @property array $json_object
 * @property string $date
 * @property Carbon $event_datetime
 * @property string $fnsku
 * @property string $asin
 * @property string $msku
 * @property string $title
 * @property string $disposition
 * @property int $starting_warehouse_balance
 * @property int $in_transit_between_warehouses
 * @property int $receipts
 * @property int $customer_shipments
 * @property int $customer_returns
 * @property int $vendor_returns
 * @property int $warehouse_transfer_in_out
 * @property int $found
 * @property int $lost
 * @property int $damaged
 * @property int $disposed
 * @property int $other_events
 * @property int $adjustments
 * @property int $ending_warehouse_balance
 * @property int $total_inventory_quantity
 * @property int $unknown_events
 * @property string $location
 * @property-read AmazonIntegrationInstance $integrationInstance
 * @property-read AmazonReport $amazonReport
 * @property-read AmazonFnskuProduct $amazonFnskuProduct
 * @property-read AmazonFbaReportInventoryLedger[]|EloquentCollection $ledgers
 * @property-read AmazonLedgerSummaryAdjustment[]|EloquentCollection $ledgerSummaryAdjustments
 * @property-read InventoryAdjustment[]|EloquentCollection $inventoryAdjustments
 * @property Carbon $created_at
 * @property ?Carbon $updated_at
 */
class AmazonFbaReportInventoryLedgerSummary extends AbstractAmazonReport implements Filterable, Sortable, DataTableModelInterface
{
    use BulkImport,
        Compoships,
        DataTableModelTrait,
        HasFactory,
        HasFilters,
        HasSort;

    protected $table = 'amazon_fba_report_inventory_ledger_summaries';

    protected $guarded = [];

    protected $casts = [
        'event_datetime' => 'datetime',
        'reconciled_at' => 'datetime',
        'json_object' => 'array',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    /**
     * @throws Exception
     */
    public function amazonReport(): BelongsTo
    {
        return $this->belongsTo(AmazonReport::class);
    }

    public function amazonFnskuProduct(): BelongsTo
    {
        return $this->belongsTo(AmazonFnskuProduct::class, [
            'integration_instance_id',
            'fnsku',
            'location',
            'disposition'
        ],
        [
            'integration_instance_id',
            'fnsku',
            'location',
            'disposition'
        ]);
    }

    public function ledgers(): \Awobaz\Compoships\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(AmazonFbaReportInventoryLedger::class, [
            'integration_instance_id',
            'date',
            'fnsku',
            'country',
            'disposition'
        ],
        [
            'integration_instance_id',
            'date',
            'fnsku',
            'location',
            'disposition'
        ]);
    }

    public function ledgerSummaryAdjustments(): HasMany
    {
        return $this->hasMany(AmazonLedgerSummaryAdjustment::class, 'amazon_fba_report_inventory_ledger_summary_id');
    }

    public function inventoryAdjustments(): HasManyThrough
    {
        return $this->hasManyThrough(
            InventoryAdjustment::class,
            AmazonLedgerSummaryAdjustment::class,
            'amazon_fba_report_inventory_ledger_summary_id',
            'id',
            'id',
            'inventory_adjustment_id'
        );
    }

    /*
    |--------------------------------------------------------------------------
    | Implementers for Filterable
    |--------------------------------------------------------------------------
    */

    public static function specialLabels(): array
    {
        return [
            'fnsku' => 'FNSKU',
        ];
    }

    public function availableColumns(): array
    {
        return [
            'id' => 'integer',
            'date' => 'date',
            'fnsku' => 'string',
            'reconciled_at' => 'datetime',
            'errorLog' => 'string',
            'asin' => 'string',
            'msku' => 'string',
            'title' => 'string',
            'disposition' => 'string',
            'starting_warehouse_balance' => 'integer',
            'in_transit_between_warehouses' => 'integer',
            'receipts' => 'integer',
            'customer_shipments' => 'integer',
            'customer_returns' => 'integer',
            'vendor_returns' => 'integer',
            'warehouse_transfer_in_out' => 'integer',
            'found' => 'integer',
            'lost' => 'integer',
            'damaged' => 'integer',
            'disposed' => 'integer',
            'other_events' => 'integer',
            'adjustments' => 'integer',
            'ending_warehouse_balance' => 'integer',
            'total_inventory_quantity' => 'integer',
            'unknown_events' => 'integer',
            'location' => 'string',
        ];
    }

    public function visibleColumns(): array
    {
        return [
            'id',
            'reconciled_at',
            'errorLog',
            'date',
            'fnsku',
            'location',
            'disposition',
            'starting_warehouse_balance',
            'in_transit_between_warehouses',
            'receipts',
            'customer_shipments',
            'customer_returns',
            'vendor_returns',
            'warehouse_transfer_in_out',
            'found',
            'lost',
            'damaged',
            'disposed',
            'other_events',
            'ending_warehouse_balance',
            'total_inventory_quantity',
            'unknown_events',
        ];
    }

    public function generalFilterableColumns(): array
    {
        return ['msku', 'fnsku', 'asin', 'title'];
    }

    /*
    |--------------------------------------------------------------------------
    | Implementers for Sortable
    |--------------------------------------------------------------------------
    */

    public function sortableColumns(): array
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeBeforeEventDate(Builder $query, $date): Builder
    {
        return $query->where('event_datetime', '<=', Carbon::parse($date));
    }

    public function scopeFnskuSort(Builder $builder): Builder
    {
        return $builder->orderBy('fnsku')
            ->orderBy('disposition')
            ->orderBy('location');
    }
}
