<?php

namespace Modules\Amazon\Entities;

use App\DataTable\Exports\DataTableExporter as Exportable;
use App\Exporters\MapsExportableFields;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Awobaz\Compoships\Compoships;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Kirschbaum\PowerJoins\PowerJoins;
use Modules\Amazon\Enums\Entities\AmazonProductFulfillmentChannelEnum;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property-read string $sku
 * @property-read string $fnsku
 * @property-read string $product_name
 * @property-read string $condition
 * @property-read string $your_price
 * @property-read string $mfn_listing_exists
 * @property-read int $mfn_fulfillable_quantity
 * @property-read string $afn_listing_exists
 * @property-read string $afn_warehouse_quantity
 * @property-read int $afn_fulfillable_quantity
 * @property-read int $afn_unsellable_quantity
 * @property-read int $afn_reserved_quantity
 * @property-read int $afn_total_quantity
 * @property-read float $per_unit_volume
 * @property-read int $afn_inbound_working_quantity
 * @property-read int $afn_inbound_shipped_quantity
 * @property-read int $afn_inbound_receiving_quantity
 * @property-read int $afn_researching_quantity
 * @property-read int $afn_reserved_future_supply
 * @property-read int $afn_future_supply_buyable
 * @property array $json_object
 * @property-read AmazonIntegrationInstance $integrationInstance
 * @property-read AmazonProduct $amazonProduct
 * @property Carbon $created_at
 * @property null|Carbon $updated_at
 * @property-read int $available_quantity
 * @property-read int $reserved_quantity
 * @property-read int $inbound_quantity
 * @property-read int $total_quantity
 */
class AmazonFbaReportInventory extends AbstractAmazonReport implements Filterable, MapsExportableFields, Sortable
{
    use BulkImport,
        Compoships,
        HasFactory,
        HasFilters,
        HasSort;

    protected $table = 'amazon_fba_report_inventory';

    protected $guarded = [];

    protected $casts = [
        'json_object' => 'array',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public static function getExportableFields(): array
    {
        return [

            'id' => Exportable::makeExportableField('id', false),
            'sku' => Exportable::makeExportableField('sku', false),
            'fnsku' => Exportable::makeExportableField('fnsku', false),
            'product_name' => Exportable::makeExportableField('product_name', false),
            'condition' => Exportable::makeExportableField('condition', false),
            'your_price' => Exportable::makeExportableField('your_price', false),
            'afn_warehouse_quantity' => Exportable::makeExportableField('afn_warehouse_quantity', false),
            'afn_fulfillable_quantity' => Exportable::makeExportableField('afn_fulfillable_quantity', false),
            'afn_unsellable_quantity' => Exportable::makeExportableField('afn_unsellable_quantity', false),
            'afn_reserved_quantity' => Exportable::makeExportableField('afn_reserved_quantity', false),
            'afn_total_quantity' => Exportable::makeExportableField('afn_total_quantity', false),
            'afn_inbound_working_quantity' => Exportable::makeExportableField('afn_inbound_working_quantity', false),
            'afn_inbound_shipped_quantity' => Exportable::makeExportableField('afn_inbound_shipped_quantity', false),
            'afn_inbound_receiving_quantity' => Exportable::makeExportableField('afn_inbound_receiving_quantity', false),
            'afn_researching_quantity' => Exportable::makeExportableField('afn_researching_quantity', false),
            'afn_reserved_future_supply' => Exportable::makeExportableField('afn_reserved_future_supply', false),
            'afn_future_supply_buyable' => Exportable::makeExportableField('afn_future_supply_buyable', false),
            'created_at' => Exportable::makeExportableField('created_at', false),
            'updated_at' => Exportable::makeExportableField('updated_at', false),
        ];
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function getAvailableQuantityAttribute(): int
    {
        return (int) $this['afn_fulfillable_quantity'];
    }

    public function getReservedQuantityAttribute(): int
    {
        return $this['afn_unsellable_quantity'] + $this['afn_reserved_quantity'];
    }

    public function getInboundQuantityAttribute(): int
    {
        return $this['afn_inbound_working_quantity'] + $this['afn_inbound_shipped_quantity'] + $this['afn_inbound_receiving_quantity'];
    }

    public function getTotalQuantityAttribute(): int
    {
        return $this->available_quantity + $this->reserved_quantity;
    }

    /*
    |--------------------------------------------------------------------------
    | Implementers for Filterable
    |--------------------------------------------------------------------------
    */

    /**
     * @throws Exception
     */
    public function amazonProduct(): BelongsTo
    {
        return $this->belongsTo(AmazonProduct::class, [
            'integration_instance_id',
            'sku',
        ], [
            'integration_instance_id',
            'seller_sku',
        ])->where('amazon_products.fulfillment_channel', '!=', AmazonProductFulfillmentChannelEnum::DEFAULT);
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function availableColumns(): array
    {
        return config('data_table.amazon.fba_report_inventory.columns');
    }

    /*
    |--------------------------------------------------------------------------
    | Implementers for Sortable
    |--------------------------------------------------------------------------
    */

    public function generalFilterableColumns(): array
    {
        return ['sku', 'fnsku', 'product_name'];
    }

    public function sortableColumns(): array
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }
}
