<?php

namespace Modules\Amazon\Database\Seeders;

use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\Store;
use App\Models\Warehouse;
use Exception;
use Illuminate\Database\Seeder;
use Illuminate\Foundation\Testing\WithFaker;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Entities\AmazonProduct;
use Throwable;

class AmazonSeeder extends Seeder
{
    use WithFaker;

    public function __construct()
    {
        $this->faker = $this->makeFaker();
    }

    /**
     * Run the database seeds.
     *
     * @throws Exception
     * @throws Throwable
     */
    public function run(): void
    {
        // Seed an Amazon Integration Instance if one doesn't exist
        if (! ($amazonIntegrationInstance = AmazonIntegrationInstance::query()->first())) {
            /** @var AmazonIntegrationInstance $amazonIntegrationInstance */
            $amazonIntegrationInstance = AmazonIntegrationInstance::factory()
                ->has(SalesChannel::factory()->recycle(Store::factory()->create(['name' => 'Amazon Store', 'company_name' => 'ABC Seller', 'email' => 'seller@abc.com'])))
                ->create();
        }

        Warehouse::factory()
            ->create([
                'name' => "FBA ($amazonIntegrationInstance->name)",
                'type' => Warehouse::TYPE_AMAZON_FBA,
                'integration_instance_id' => $amazonIntegrationInstance->id,
            ]);

        // Seed some Amazon Products, ProductListings, and Products
        AmazonProduct::factory(10)
            ->has(
                ProductListing::factory(1, [
                    'sales_channel_id' => $amazonIntegrationInstance->salesChannel->id,
                    // This is required because the has method doesn't handle the complex relationship
                    'document_type' => AmazonProduct::class,
                ])
                    ->hasProduct()
            )
            ->afterCreating(function (AmazonProduct $product, $count) {
                $listingSku = $product->productListing->listing_sku;
                $product->update(['json_object->seller_sku' => $listingSku]);
            })
            ->create(['integration_instance_id' => $amazonIntegrationInstance->id]);

        Product::query()->each(function (Product $product) {
            $product->setInitialInventory(1, 500, $this->faker->randomFloat(2, 1, 100));
        });

        // Seed some Amazon Orders and Order Items
        AmazonOrder::factory()
            ->times(5)
            ->hasAmazonOrderItems(random_int(1, 5))
            ->state(function (array $attributes) {
                return [
                    'json_object->FulfillmentChannel' => 'AFN',
                ];
            })
            ->create();

        AmazonOrder::factory()
            ->times(5)
            ->hasAmazonOrderItems(random_int(1, 5))
            ->state(function (array $attributes) {
                return [
                    'json_object->FulfillmentChannel' => 'MFN',
                ];
            })
            ->create();
    }
}
