<?php

namespace Modules\Amazon\Database\Factories;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\Factory;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Enums\Entities\OrderStatusEnum;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\Modules\Amazon\Entities\AmazonOrder>
 */
class AmazonOrderFactory extends Factory
{
    /**
     * The name of the factory's corresponding model.
     *
     * @var string
     */
    protected $model = AmazonOrder::class;

    /**
     * Define the model's default state.
     */
    public function definition(): array
    {
        /** @var AmazonIntegrationInstance $amazonIntegrationInstance */
        $amazonIntegrationInstance = AmazonIntegrationInstance::query()->first();

        $shipServiceLevel = $this->faker->randomElement(['Standard', 'Expedited', 'Priority']);
        $fulfillmentChannel = $this->faker->randomElement(['AFN', 'MFN']);

        return [
            'integration_instance_id' => $amazonIntegrationInstance->id,
            'json_object' => [
                'AmazonOrderId' => sprintf('%03d-%07d-%07d',
                    $this->faker->numberBetween(100, 999),
                    $this->faker->numberBetween(1000000, 9999999),
                    $this->faker->unique()->numberBetween(1000000, 9999999)
                ),
                'SellerOrderId' => $this->faker->unique()->numberBetween(),
                'PurchaseDate' => Carbon::parse($this->faker->dateTimeBetween('-1 year'))->toIso8601String(),
                'PurchaseDateUtc' => Carbon::parse($this->faker->dateTimeBetween('-1 year'))->toDateTimeString(),
                'LastUpdateDate' => Carbon::parse($this->faker->date())->toIso8601String(),
                'LastUpdateDateUtc' => Carbon::parse($this->faker->date())->toDateTimeString(),
                'EarliestShipDate' => Carbon::parse($this->faker->date())->toIso8601String(),
                'EarliestShipDateUtc' => Carbon::parse($this->faker->date())->toDateTimeString(),
                'LatestShipDate' => Carbon::parse($this->faker->date())->toIso8601String(),
                'LatestShipDateUtc' => Carbon::parse($this->faker->date())->toDateTimeString(),
                'OrderStatus' => $this->faker->randomElement(OrderStatusEnum::values()),
                'FulfillmentChannel' => $fulfillmentChannel,
                'SalesChannel' => $fulfillmentChannel,
                'OrderTotal' => [
                    'CurrencyCode' => 'USD', //$this->faker->currencyCode(), //TODO: Specify possible codes here
                    'Amount' => $this->faker->randomFloat(2, 0, 1000),
                ],
                'ShipServiceLevel' => $shipServiceLevel,
                'ShipmentServiceLevelCategory' => $shipServiceLevel,
                'OrderCurrency' => 'USD',
            ],
        ];
    }

    public function configure()
    {
        // Handle PII
        return $this->afterCreating(function (AmazonOrder $amazonOrder) {
            $buyerInfo = [
                'BuyerEmail' => $this->faker->unique()->userName().'@marketplace.amazon.com',
            ];

            $shippingAddress = [
                'CountryCode' => 'US',
                'Country' => 'United States',
                'City' => $this->faker->city(),
                'PostalCode' => $this->faker->postcode(),
                'StateOrRegion' => $this->faker->stateAbbr(),
            ];

            if ($amazonOrder->json_object['FulfillmentChannel'] == 'MFN') {
                $buyerInfo['BuyerName'] = $this->faker->unique()->name();
                $shippingAddress['Name'] = encryptValue($this->faker->unique()->name());
                $shippingAddress['AddressLine1'] = encryptValue($this->faker->streetAddress());
                $shippingAddress['Phone'] = encryptValue($this->faker->phoneNumber());
            }

            $json_object = $amazonOrder->json_object;
            $json_object['ShippingAddress'] = $shippingAddress;
            $json_object['BuyerInfo'] = encryptArray($buyerInfo);

            $amazonOrder->json_object = $json_object;
            $amazonOrder->save();
        });
    }
}
