<?php

namespace Modules\Amazon\Console;

use App\Helpers;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Modules\Amazon\Actions\GenerateAmazonGapReportRequests;
use Modules\Amazon\Actions\RequestAmazonReport;
use Modules\Amazon\Data\RequestAmazonReportData;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Spatie\LaravelData\Optional;
use Throwable;

class RequestAmazonReportCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'amazon:reports:request
                            {integrationInstanceId? : The Integration Instance ID for Amazon}
                            {reportType? : The Type of Report (i.e.: GET_MERCHANT_LISTINGS_ALL_DATA)}
                            {--d|days= : Days of data}
                            {--s|startDate= : Report start date (2019-10-31)}
                            {--e|endDate= : Report end date (2019-10-31)}
                            {--o|options=* : Other report options key value pair with = separator}
                            {--a|available : Return list of available reports}
                            {--x|dry : Dry run, lists the dates that would be requested}
                            {--g|gap : Generate multiple requests for all gaps in the reports}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create an Amazon Report Request.';

    /**
     * Execute the console command.
     *
     * @throws Throwable
     */
    public function handle(): void
    {
        $arguments = $this->data();

        if ($this->option('available')) {
            AmazonIntegrationInstance::query()
                ->select(['integration_instances.id', 'integrations.name'])
                ->join('integrations', 'integrations.id', 'integration_instances.integration_id')
                ->each(function ($instance) {
                    $this->line("$instance->id: $instance->name");
                });
            collect(AmazonReportTypeEnum::values())->each(function ($type) {
                $this->line($type);
            });

            return;
        }

        if (! isset($arguments['integrationInstanceId'])) {
            $this->error('Missing integrationInstanceId');
        }

        if (! isset($arguments['reportType'])) {
            $this->error('Missing reportType');
        }

        $amazonIntegrationInstance = AmazonIntegrationInstance::find($arguments['integrationInstanceId']);

        $amazonReportRequests = collect();

        if ($this->option('gap')) {
            $amazonReportRequests = (new GenerateAmazonGapReportRequests($amazonIntegrationInstance, AmazonReportTypeEnum::tryFrom($arguments['reportType'])))->__invoke();
        } else
        {
            $requestData = RequestAmazonReportData::from([
                'amazonIntegrationInstance' => $amazonIntegrationInstance,
                'report_type' => AmazonReportTypeEnum::tryFrom($arguments['reportType']),
                'createImmediately' => false,
            ]);

            $startDate = $this->option('days') ?
                Carbon::now('UTC')->subDays($this->option('days') + 1) :
                ($this->option('startDate') ? Helpers::dateLocalToUtc($this->option('startDate')) : null);
            $endDate = $this->option('days') ?
                Carbon::now('UTC')->subDay() :
                ($this->option('endDate') ? Helpers::dateLocalToUtc($this->option('endDate')) : null);

            if ($startDate) {
                $requestData->data_start_date = $startDate;
            }
            if ($endDate) {
                $requestData->data_end_date = $endDate;
            }
            if ($this->option('options')) {
                $requestData->options = Helpers::array2multidimensional($this->option('options'), '=');
            }

            $amazonReportRequests->push($requestData);
        }

        if ($this->option('dry')) {
            $this->info('Dry run for '.$arguments['reportType']);
        }
        $amazonReportRequests->each(function (RequestAmazonReportData $requestData) use ($arguments) {

            if ($this->option('dry')) {
                $this->info( 'start date: '.($requestData->data_start_date ?? 'unspecified') .', end date: ' . ($requestData->data_end_date ?? 'unspecified'));
            } else
            {
                $responseData = (new RequestAmazonReport($requestData))->handle();

                if (
                    empty($responseData->createdRequests) &&
                    empty($responseData->createdReports)
                ) {
                    $this->warn('Existing report in queue for '.$arguments['reportType'].', 
            so no action needed (Instance: '.$arguments['integrationInstanceId'].')');
                }
            }

        });
    }

    private function data(): array
    {
        return [
            'integrationInstanceId' => $this->argument('integrationInstanceId') ?? $this->choice('Integration Instance ID', AmazonIntegrationInstance::query()->pluck('id')->toArray()),
            'reportType' => $this->argument('reportType') ?? $this->choice('Report Type', AmazonReportTypeEnum::REQUESTABLE_REPORTS),
        ];
    }
}
