<?php

namespace Modules\Amazon\ApiDataTransferObjects;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Models\SalesChannel;
use Illuminate\Support\Collection;
use Modules\Amazon\Data\AmazonCreateOrderFulfillmentFeedData;
use Spatie\LaravelData\Attributes\DataCollectionOf;

class AmazonFulfillOrdersAdt implements ApiDataTransformerInterface
{
    public function __construct(
        #[DataCollectionOf(AmazonCreateOrderFulfillmentFeedData::class)] public $amazonCreateOrderFulfillmentFeedDtoCollection,
        public Collection $messages = new Collection(),
    ) {
    }

    public function buildMessages(): Collection
    {
        $carrierCodes = config('amazon.carrier_codes');
        $this->amazonCreateOrderFulfillmentFeedDtoCollection->each(function (AmazonCreateOrderFulfillmentFeedData $amazonCreateOrderFulfillmentFeedDto, $index) use ($carrierCodes) {
            $items = [];
            collect($amazonCreateOrderFulfillmentFeedDto->Items)->each(function (array $item) use (&$items) {
                $items[] = [
                    'AmazonOrderItemCode' => $item['AmazonOrderItemCode'],
                    'MerchantFulfillmentItemID' => $item['MerchantFulfillmentItemID'],
                    'Quantity' => $item['Quantity'],
                ];
            });
            $data = [
                'MessageID' => $index + 1,
                'OrderFulfillment' => [
                    'AmazonOrderID' => $amazonCreateOrderFulfillmentFeedDto->AmazonOrderID,
                    'MerchantFulfillmentID' => $amazonCreateOrderFulfillmentFeedDto->MerchantFulfillmentID,
                    'FulfillmentDate' => $amazonCreateOrderFulfillmentFeedDto->FulfillmentDate,
                    'FulfillmentData' => [
                        'CarrierCode' => in_array($amazonCreateOrderFulfillmentFeedDto->FulfillmentData_CarrierName, $carrierCodes) ? $amazonCreateOrderFulfillmentFeedDto->FulfillmentData_CarrierName : 'Other',
                        // 'CarrierName' will be conditionally inserted here
                        'ShippingMethod' => SalesChannel::UNSPECIFIED_SHIPPING_METHOD,
                        'ShipperTrackingNumber' => $amazonCreateOrderFulfillmentFeedDto->FulfillmentData_ShipperTrackingNumber,
                    ],
                    'Item' => $items,
                ],
            ];

            // Insert 'CarrierName' after 'CarrierCode' if CarrierCode is 'Other'
            if ($data['OrderFulfillment']['FulfillmentData']['CarrierCode'] === 'Other') {
                $data['OrderFulfillment']['FulfillmentData'] = array_merge(
                    array_slice($data['OrderFulfillment']['FulfillmentData'], 0, 1), // Elements before 'CarrierName'
                    ['CarrierName' => $amazonCreateOrderFulfillmentFeedDto->FulfillmentData_CarrierName], // The 'CarrierName' element
                    array_slice($data['OrderFulfillment']['FulfillmentData'], 1)  // Elements after 'CarrierCode'
                );
            }

            $this->messages->push($data);
        });

        return $this->messages;
    }

    public function transform(): array
    {
        $this->buildMessages();

        $params = [
            'Header' => [
                'DocumentVersion' => '1.01',
                //'MerchantIdentifier' => $this->merchantId, // Delegated to the client
            ],
            'MessageType' => 'OrderFulfillment',
            'Message' => $this->messages->toArray(),
        ];

        return array_filter($params, fn ($param) => ! is_null($param));
    }
}
