<?php

namespace Modules\Amazon\Actions\LedgerSummaryActions;

use App\Exceptions\InsufficientStockException;
use Exception;
use Illuminate\Support\Collection;
use Modules\Amazon\Data\AmazonLedgerSummaryAdjustmentData;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedgerSummary;
use Modules\Amazon\Enums\Entities\AmazonLedgerSummaryAdjustmentTypeEnum;
use Modules\Amazon\Repositories\AmazonLedgerSummaryRepository;
use Throwable;

class AmazonLedgerSummaryTotalInventoryDiscrepancyHandler
{
    public function __construct(private readonly AmazonLedgerSummaryRepository $summaries) {}

    /**
     * @throws InsufficientStockException
     * @throws Throwable
     */
    public function handle(AmazonFbaReportInventoryLedgerSummary $ledgerSummary, Collection $amazonLedgerSummaryAdjustmentDataCollection): void
    {
        if ($ledgerSummary->ledgerSummaryAdjustments->where('type', AmazonLedgerSummaryAdjustmentTypeEnum::TotalInventoryAdjustment)->isNotEmpty()) {
            throw new Exception('Unexpected attempt to adjust total inventory quantity for the same ledger summary twice');
        }

        $previousLedgerSummary = $this->summaries->getPreviousSummary($ledgerSummary);

        $calculatedTotalInventoryQuantity = ($previousLedgerSummary->total_inventory_quantity ?? 0) + $ledgerSummary->ledgers->sum('quantity') + $ledgerSummary->unknown_events;
        $discrepancyQuantity = $ledgerSummary->total_inventory_quantity - $calculatedTotalInventoryQuantity;

        if ($discrepancyQuantity == 0) {
            return;
        }

        $amazonLedgerSummaryAdjustmentDataCollection->push(AmazonLedgerSummaryAdjustmentData::from([
            'amazon_fba_report_inventory_ledger_summary_id' => $ledgerSummary->id,
            'type' => AmazonLedgerSummaryAdjustmentTypeEnum::TotalInventoryAdjustment,
            'inventory_adjustment_id' => app(AmazonCreateInventoryAdjustmentFromLedgerSummary::class)->handle(
                $ledgerSummary,
                $discrepancyQuantity,
                "FNSKU $ledgerSummary->fnsku " . friendlyName(AmazonLedgerSummaryAdjustmentTypeEnum::TotalInventoryAdjustment->value),
            )->id,
        ]));
    }
}