<?php

namespace Modules\Amazon\Actions\LedgerActions\SkuModels;

use App\Exceptions\InsufficientStockException;
use App\Repositories\ShippingCarrierRepository;
use App\Repositories\WarehouseTransferRepository;
use App\Services\InventoryManagement\InventoryManager;
use Carbon\Carbon;
use Modules\Amazon\Abstractions\AmazonLedgerHandlerInterface;
use Modules\Amazon\Actions\LedgerActions\AmazonCreateInventoryAdjustmentFromLedger;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Entities\AmazonFbaReportRemovalShipment;
use Modules\Amazon\Enums\Entities\FbaRemovalOrderTypeEnum;
use Throwable;

/*
 * A Warehouse Transfer Shipment Line must be created, but the Warehouse Transfer must exist
 * (created by User intervention based on Removal Orders Report).  A detail report for Removal
 * Order Shipments must exist (detail report) as well
 *
 * If no detail report exists (despite having been requested), can create an adjustment instead
 */
class AmazonProcessWarehouseTransferShipmentLine implements AmazonLedgerHandlerInterface
{
    public function __construct(
        private readonly WarehouseTransferRepository $warehouseTransfers,
        private readonly ShippingCarrierRepository $carriers,
    ) {}

    /**
     * @throws Throwable
     * @throws InsufficientStockException
     */
    public function handle(AmazonFbaReportInventoryLedger $ledger): void
    {
        $product = $ledger->amazonFnskuProduct->product;

        /** @var AmazonFbaReportRemovalShipment $detailReport */
        if ($detailReport = $ledger->details?->first()) {
            customlog('amazon', 'Warehouse Transfer Shipment Ledger Detail Report Found.');

            if ($detailReport->removal_order_type != FbaRemovalOrderTypeEnum::Return) {
                customlog('amazon', 'Warehouse Transfer Shipment Ledger is not for a return, it is for a '.$detailReport->removal_order_type->value.', so creating an adjustment instead.');
                $notes = "Removal Shipment Ledger (ID: $ledger->id) of type {$detailReport->removal_order_type->value}.";
                app(AmazonCreateInventoryAdjustmentFromLedger::class)->handle($ledger, $notes);

                return;
            }

            // Check if Warehouse Transfer exists
            if (! $warehouseTransfer = $this->warehouseTransfers->getFromWarehouseTransferNumber($detailReport->order_id)) {
                customlog('amazon', 'Warehouse Transfer Shipment Ledger does not have a Warehouse Transfer match for '.$detailReport->order_id.'.');
                $ledger->errorLog = 'Missing Warehouse Transfer';
                $ledger->last_reconciliation_attempted_at = Carbon::now();
                $ledger->save();

                return;
                //                                $notes = "Warehouse Transfer for Removal Shipment Ledger (ID: $amazonFbaReportInventoryLedger->id) could not be determined due to no Warehouse Transfer match.";
                //                                $this->createInventoryAdjustmentFromLedger($amazonFbaReportInventoryLedger, $product, $notes);
                //
                //                                return;
            }

            $shippingMethod = $this->carriers->findOrCreateMethodForCarrier($detailReport->carrier);

            if (! $warehouseTransferShipment = $this->warehouseTransfers->getSimilarWarehouseTransferShipment(
                $warehouseTransfer,
                Carbon::parse($ledger->event_datetime),
                $shippingMethod,
                $detailReport->tracking_number
            )) {
                customlog('amazon', 'Warehouse Transfer Shipment '.$detailReport->order_id.' does not have a shipment for '.$ledger->date.', so creating.');
                // Create shipment
                $warehouseTransferShipment = $this->warehouseTransfers->createTransferShipment($warehouseTransfer, [
                    'shipment_date' => Carbon::parse($ledger->event_datetime)->format('Y-m-d H:i:s'),
                    'shipping_method_id' => $shippingMethod->id,
                    'tracking_number' => $detailReport->tracking_number,
                ]);
            }

            /** @var AmazonFbaReportRemovalShipment $amazonFbaReportRemovalShipment */
            $amazonFbaReportRemovalShipment = $detailReport;
            $warehouseTransferShipmentLine = $this->warehouseTransfers->createTransferShipmentLine(
                $warehouseTransferShipment,
                $amazonFbaReportRemovalShipment->amazonFbaReportRemovalOrder->skuLink->id,
                -$ledger->quantity
            );

            InventoryManager::with(
                $warehouseTransfer->from_warehouse_id,
                $product
            )->takeFromStock($warehouseTransferShipmentLine->quantity, $warehouseTransferShipmentLine);

            $ledger->skuLink()->associate($warehouseTransferShipmentLine);
            $ledger->last_reconciliation_attempted_at = Carbon::now();
            $ledger->reconciled_at = Carbon::now();
            $ledger->errorLog = null;
            $ledger->blocked_by_ledger_id = null;
            $ledger->save();

            $ledger->amazonFnskuProduct->reconciled_quantity += $ledger->quantity;
            $ledger->amazonFnskuProduct->save();

        } else {
            customlog('amazon', 'Warehouse Transfer Shipment for Shipment Ledger (ID: '.$ledger->id.') could not be determined due to no detail report match.');
            $ledger->errorLog = 'Pending Detail Report Processing';
            $ledger->last_reconciliation_attempted_at = Carbon::now();
            $ledger->save();
        }
    }
}