<?php

namespace Modules\Amazon\Actions\LedgerActions\SkuModels;

use App\Data\WarehouseTransferReceiptData;
use App\Data\WarehouseTransferReceiptProductData;
use App\Exceptions\InsufficientStockException;
use App\Managers\WarehouseTransferManager;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferShipment;
use App\Models\WarehouseTransferShipmentReceipt;
use App\Repositories\PurchaseOrderShipmentReceiptRepository;
use App\Repositories\WarehouseTransferRepository;
use App\Services\PurchaseOrder\ShipmentManager;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Abstractions\AmazonLedgerHandlerInterface;
use Modules\Amazon\Actions\LedgerActions\AmazonCreateInventoryAdjustmentFromLedger;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Managers\AmazonInboundManager;
use Modules\Amazon\Repositories\AmazonFbaInboundShipmentRepository;
use Modules\Amazon\Repositories\AmazonNewFbaInboundShipmentRepository;
use Throwable;

class AmazonProcessWarehouseTransferOrPurchaseOrderShipmentReceipt implements AmazonLedgerHandlerInterface
{
    public function __construct(
        private readonly AmazonFbaInboundShipmentRepository $inbounds,
        private readonly AmazonNewFbaInboundShipmentRepository $newInbounds,
        private readonly WarehouseTransferRepository $warehouseTransfers,
        private readonly WarehouseTransferManager $warehouseTransferProcessor,
        private readonly PurchaseOrderShipmentReceiptRepository $purchaseOrderShipmentReceipts,
        private readonly ShipmentManager $purchaseOrderShipmentManager
    )
    {
    }

    /**
     * @throws Throwable
     * @throws InsufficientStockException
     */
    public function handle(AmazonFbaReportInventoryLedger $ledger): void
    {
        /*
         * A Warehouse Transfer Receipt or Purchase Order Shipment Receipt must be created, but the parent
         * model for each must exist and can be located using the ledger reference_id to find the inbound
         * and then the inbound to find the sku model.
         *
         */

        $product = $ledger->amazonFnskuProduct->product;

        $inboundShipment = $this->newInbounds->getFromShipmentConfirmationId(
            $ledger->integrationInstance,
            $ledger->reference_id
        ) ?? $this->inbounds->getFromShipmentId(
            $ledger->integrationInstance,
            $ledger->reference_id
        );

        if (! $inboundShipment) {
            // Attempt to first download inbound shipment from Amazon
            // Can't download new inbound shipments this way because need plan id, so just trying old
            if (! $inboundShipment = (new AmazonInboundManager($ledger->integrationInstance))->getShipment($ledger->reference_id)) {
                customlog('amazon', 'Amazon FBA Inbound Shipment not found '.$ledger->reference_id);
                // If we get here, we have a new amazon order, and we can create the sales order now
                // For now, we are going to halt processing and update the errorLog, so that the scenario
                // can be analyzed
                $ledger->errorLog = 'Missing Inbound Shipment'; // Maybe make this error from api call?
                $ledger->last_reconciliation_attempted_at = Carbon::now();
                $ledger->save();

                return;
            }

            // If we get here, the inbound shipment would have been downloaded and the process will error at the next step but with a different error code "Inbound Shipment Not Processed Yet in SKUio"
            // User has the opportunity to mark an inbound shipment as sent before initial count, which would process the ledger receipts for it as adjustments.  Then next time the adjustment will be made
            // in lieu of the warehouse transfer shipment receipt or purchase order shipment receipt
        }

        if ($inboundShipment->is_before_initial_count) {
            customlog('amazon', 'Inbound Shipment '.$inboundShipment->ShipmentId.' was marked as sent before initial count, so processing as an adjustment instead of Warehouse Transfer or Purchase Order Shipment Receipt.');
            $notes = "Receipt Ledger (ID: $ledger->id) was received inbound shipment $inboundShipment->ShipmentId which was marked by User as sent before initial count.";
            app(AmazonCreateInventoryAdjustmentFromLedger::class)->handle($ledger, $notes);

            return;
        }

        /** @var Model $skuModel */
        if (! $skuModel = $inboundShipment->skuLink) {
            customlog('amazon', 'Inbound Shipment Not Processed Yet in SKUio '.$inboundShipment->ShipmentId);
            $ledger->errorLog = 'Inbound Shipment Not Processed Yet in SKUio';
            $ledger->last_reconciliation_attempted_at = Carbon::now();
            $ledger->save();

            return;
        }
        customlog('amazon', 'Processing for inbound shipment '.$inboundShipment->ShipmentId.' found SKU model '.get_class($skuModel).' with ID '.$skuModel->id.' for Receipt Ledger (ID: '.$ledger->id.')');
        // Warehouse Transfer or Purchase Order found
        switch (get_class($skuModel)) {
            case WarehouseTransfer::class:
                /** @var WarehouseTransfer $warehouseTransfer */
                $warehouseTransfer = $skuModel;

                /*
                 * TODO: Is there potential of multiple warehouse transfer shipments?
                 *  This is still being designed within the Inbound flow.  For now let's assume one.
                 */

                /** @var WarehouseTransferShipment $warehouseTransferShipment */
                if (! $warehouseTransferShipment = $warehouseTransfer->shipments()->first()) {
                    throw new Exception('Warehouse Transfer Shipment for Receipt Ledger (ID: '.$ledger->id.') does not exist even though the warehouse transfer '.$warehouseTransfer->id.' does');
                }

                /** @var WarehouseTransferShipmentReceipt $warehouseTransferShipmentReceipt */
                if ($warehouseTransferShipmentReceipt = $this->warehouseTransfers->getWarehouseTransferShipmentReceiptForDate($warehouseTransferShipment, Carbon::parse($ledger->event_datetime))
                ) {
                    $warehouseTransferLine = $warehouseTransfer->warehouseTransferLines()->where('product_id', $product->id)->firstOrFail();

                    $productData = [
                        'id' => $product->id,
                        'quantity' => $ledger->quantity,
                    ];

                    // Receive shipment for line
                    $link = $this->warehouseTransferProcessor->receiveShipmentForTransferLine($warehouseTransferLine, $warehouseTransferShipmentReceipt, $productData);
                } else {
                    /*
                     * Using existing method.  Making the assumption this is working correctly.  If
                     * not, may need to refactor.
                     */
                    $warehouseTransferReceipt = $this->warehouseTransferProcessor->receiveShipment($warehouseTransfer, WarehouseTransferReceiptData::from([
                        'shipment_id' => $warehouseTransferShipment->id,
                        'receipt_date' => Carbon::parse($ledger->event_datetime),
                        // A single product is received from each ledger line
                        'products' => WarehouseTransferReceiptProductData::collection([WarehouseTransferReceiptProductData::from([
                            'id' => $product->id,
                            'quantity' => $ledger->quantity,
                        ])]),
                    ]));

                    if ($warehouseTransferReceipt instanceof  WarehouseTransfer){
                        // We link the ledger to the created negative inventory adjustment.
                        $link = $warehouseTransferReceipt->warehouseTransferLines()
                            ->where('product_id', $product->id)
                            ->firstOrFail()
                            ->adjustments()
                                ->latest()
                                ->where('quantity', $ledger->quantity)
                                ->firstOrFail();

                        app(WarehouseTransferManager::class)->setTransferReceiptStatus($warehouseTransferReceipt);
                    } else {
                        $link = $warehouseTransferReceipt->receiptLines()
                            ->where('quantity', $ledger->quantity)
                            ->latest()
                            ->first();
                    }
                }

                $ledger->skuLink()->associate($link);
                $ledger->last_reconciliation_attempted_at = Carbon::now();
                $ledger->reconciled_at = Carbon::now();
                $ledger->errorLog = null;
                $ledger->blocked_by_ledger_id = null;
                $ledger->save();

                $ledger->amazonFnskuProduct->reconciled_quantity += $ledger->quantity;
                $ledger->amazonFnskuProduct->save();

                break;
            case PurchaseOrder::class:
                /** @var PurchaseOrder $purchaseOrder */
                $purchaseOrder = $skuModel;

                /** @var PurchaseOrderLine $purchaseOrderLine */
                $purchaseOrderLine = $purchaseOrder->purchaseOrderLines()->where('product_id', $product->id)->firstOrFail();

                $productId = $purchaseOrderLine->product_id;

                // Add to existing purchase order receipt if one for the same date already exists
                $purchaseOrderReceipt = $this->purchaseOrderShipmentReceipts->findForDate($purchaseOrder, Carbon::parse($ledger->event_datetime));
                if ($purchaseOrderReceipt && $ledger->quantity >= 0)
                {
                    $this->purchaseOrderShipmentManager->addLinesToExistingShipmentReceipt($purchaseOrderReceipt, [
                        [
                            'quantity' => $ledger->quantity,
                            'purchase_order_line_id' => $purchaseOrderLine->id,
                        ],
                    ]);
                } else {
                    $purchaseOrderReceipt = DB::transaction(function () use ($ledger, $purchaseOrder, $purchaseOrderLine) {
                        return $this->purchaseOrderShipmentManager->receiveShipment([
                            'purchase_order_id' => $purchaseOrder->id,
                            'received_at' => Carbon::parse($ledger->event_datetime, $ledger->integrationInstance->getTimezone())
                                ->setTimezone('UTC')
                                ->format('Y-m-d H:i:s'),
                            'warehouse_id' => $ledger->integrationInstance->warehouse->id,
                            'shipment_date' => Carbon::parse($ledger->event_datetime. $ledger->integrationInstance->getTimezone())
                                ->setTimezone('UTC')
                                ->format('Y-m-d H:i:s'),
                            'receipt_lines' => [
                                [
                                    'quantity' => $ledger->quantity,
                                    // TODO: This could fail if duplicate lines on a PO for some product
                                    'purchase_order_line_id' => $purchaseOrderLine->id,
                                ],
                            ],
                        ]);
                    });
                }

                $link = $purchaseOrderReceipt instanceof PurchaseOrderShipmentReceipt ?
                    $purchaseOrderReceipt->purchaseOrderShipmentReceiptLines()
                        ->where('purchase_order_line_id', $purchaseOrderLine->id)
                        ->where('quantity', $ledger->quantity)
                        ->latest()
                        ->first() :
                    $purchaseOrderLine->adjustments()
                        ->where('quantity', $ledger->quantity)
                        ->where('product_id', $productId)
                        ->orderByDesc('created_at')
                        ->latest()
                        ->firstOrFail();

                $ledger->skuLink()->associate($link);
                $ledger->last_reconciliation_attempted_at = Carbon::now();
                $ledger->reconciled_at = Carbon::now();
                $ledger->errorLog = null;
                $ledger->blocked_by_ledger_id = null;
                $ledger->save();

                $ledger->amazonFnskuProduct->reconciled_quantity += $ledger->quantity;
                $ledger->amazonFnskuProduct->save();

                break;
            default:
                throw new Exception('Unexpected SKU Model Type');
        }

    }
}