<?php

namespace Modules\Amazon\Actions\LedgerActions\SkuModels;

use App\Exceptions\ApiException;
use App\Exceptions\InsufficientStockException;
use App\Managers\ProductInventoryManager;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Repositories\SalesOrderFulfillmentRepository;
use App\Services\InventoryManagement\InventoryDeficiencyActionType;
use App\Services\InventoryManagement\InventoryManager;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Arr;
use Modules\Amazon\Abstractions\AmazonLedgerHandlerInterface;
use Modules\Amazon\Actions\LedgerActions\AmazonCreateInventoryAdjustmentFromLedger;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Entities\AmazonFbaReportShipment;
use Modules\Amazon\Entities\AmazonFulfillmentOrder;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Managers\AmazonOrderManager;
use Modules\Amazon\Repositories\AmazonOrderItemRepository;
use Modules\Amazon\Repositories\AmazonOrderRepository;
use Modules\Amazon\Repositories\AmazonOutboundFulfillmentRepository;
use Throwable;

class AmazonProcessSalesOrderFulfillmentLine implements AmazonLedgerHandlerInterface
{
    public function __construct(
        private readonly AmazonOutboundFulfillmentRepository $fulfillmentOrders,
        private readonly AmazonOrderRepository $orders,
        private readonly AmazonOrderItemRepository $orderItems,
        private readonly SalesOrderFulfillmentRepository $salesOrderFulfillments,
    )
    {
    }

    /**
     * @throws Throwable
     * @throws InsufficientStockException
     */
    public function handle(AmazonFbaReportInventoryLedger $ledger): void
    {
        $product = $ledger->amazonFnskuProduct->product;

        /** @var AmazonFbaReportShipment $detailReport */
        if ($detailReport = $ledger->details?->first()) {
            //customlog('amazon', 'Detail Report exists.  Attempting to link to Sales Order');

            if ($detailReport->sales_channel == 'Non-Amazon')
            {
                /** @var AmazonFulfillmentOrder $fulfillmentOrder */
                $fulfillmentOrder = $this->fulfillmentOrders->getForValue($detailReport->merchant_order_id, 'sellerFulfillmentOrderId', AmazonFulfillmentOrder::class);

                if (! $fulfillmentOrder) {
                    // This could be an order that was created before the start date of the integration.  We'll create an adjustment here instead
                    customlog('amazon', 'Fulfillment Order for Shipment Ledger (ID: $ledger->id) could not be determined due to no fulfillment order match.');
                    $notes = "Fulfillment Order for Shipment Ledger (ID: $ledger->id) could not be determined due to no fulfillment order match.";
                    app(AmazonCreateInventoryAdjustmentFromLedger::class)->handle($ledger, $notes);

                    return;
                }

                $salesOrderFulfillment = $fulfillmentOrder->salesOrderFulfillment;

                $salesOrderFulfillmentLine = $salesOrderFulfillment->salesOrderFulfillmentLines->where('salesOrderLine.product_id', $product->id)->first();

                if (! $salesOrderFulfillmentLine) {
                    customlog('amazon', 'Sales order fulfillment line for Shipment Ledger (ID: '.$ledger->id.') could not be determined due to no sales order fulfillment line match.');
                    $ledger->errorLog = 'Missing Sales Order Fulfillment Line';
                    $ledger->last_reconciliation_attempted_at = Carbon::now();
                    $ledger->save();

                    return;
                }
            }
            else
            {
                // Look up Amazon Order from detail report
                /** @var AmazonOrder $amazonOrder */
                if (! $amazonOrder = $this->orders->getOrderFromId(
                    $ledger->integrationInstance,
                    $detailReport->amazon_order_id
                )) {
                    customlog('amazon', 'Amazon Order '.$detailReport->amazon_order_id.' does not exist.  Attempting to get');
                    // Attempt to first download sales order from Amazon

                    $delay = 5;
                    do {
                        try {
                            $amazonOrder = (new AmazonOrderManager($ledger->integrationInstance))->getOrder($detailReport->amazon_order_id);
                            $code = null;
                        } catch (ApiException $e) {
                            $code = $e->getCode();
                            customlog('amazon', $code.' error.  Waiting '.$delay.' seconds and trying again');
                            sleep($delay);
                            $delay *= 2;
                        }
                    } while ($code == 429);

                    if (! $amazonOrder) {
                        customlog('amazon', 'Amazon Order '.$detailReport->amazon_order_id.' does not exist and cannot be downloaded');
                        $ledger->errorLog = 'Missing Amazon Order';
                        $ledger->last_reconciliation_attempted_at = Carbon::now();
                        $ledger->save();

                        return;
                    }
                    // If we get here, we have a new amazon order, and we can create the sales order now
                    (new AmazonOrderManager($ledger->integrationInstance))->createSkuOrders([$amazonOrder->id], false, false);
                }

                /** @var SalesOrder $salesOrder */
                if (! $salesOrder = $amazonOrder->salesOrder) {
                    // Look up Sales Order from Amazon Order
                    /** @var SalesOrder $salesOrder */
                    (new AmazonOrderManager($ledger->integrationInstance))->createSkuOrders(Arr::wrap($amazonOrder->id));
                    customlog('amazon', 'No Sales Order found for Amazon Order ID: '.$amazonOrder->id.', attempting to create...');
                    $amazonOrder->refresh();
                    if (! $salesOrder = $amazonOrder->salesOrder) {
                        // If ledger is referring to a sales order that is before the start date, still create it
                        if ($amazonOrder->PurchaseDateUtc->lt($ledger->integrationInstance->integration_settings['start_date'])) {
                            (new AmazonOrderManager($ledger->integrationInstance))->createSkuOrders([$amazonOrder->id], false, false);
                            $amazonOrder->refresh();
                            if (! $salesOrder = $amazonOrder->salesOrder) {
                                $ledger->errorLog = 'Missing Sales Order';
                                $ledger->last_reconciliation_attempted_at = Carbon::now();
                                $ledger->save();

                                return; // Halt audit trail processing for this product
                            }
                        } else {
                            $ledger->errorLog = 'Missing Sales Order';
                            $ledger->last_reconciliation_attempted_at = Carbon::now();
                            $ledger->save();

                            return; // Halt audit trail processing for this product
                        }
                    }
                }

                $amazonOrderItem = $this->orderItems->getFromOrderItemId(
                    $amazonOrder->id,
                    $detailReport->merchant_order_item_id != '' ? $detailReport->merchant_order_item_id : $detailReport->amazon_order_item_id
                );

                /*
                 * Get Sales Order Line from Amazon Order Item
                 * If a sales order line can't be determined, we are forced to create an adjustment instead
                 */
                if (! $salesOrderLine = $amazonOrderItem->salesOrderLine) {
                    // We want to throw an exception here to analyze how this could happen before deciding to handle it with an adjustment
                    throw new Exception('Sales order for Shipment Ledger (ID: '.$ledger->id.') could not be determined due to no sales order line match.');
                    //                                $notes = "Sales order for Shipment Ledger (ID: $amazonFbaReportInventoryLedger->id) could not be determined due to no sales order line match.";
                    //                                $this->createInventoryAdjustmentFromLedger($amazonFbaReportInventoryLedger, $product, $notes);
                    //
                    //                                return;
                }

                if (! $salesOrderLine->product_id) {
                    $ledger->errorLog = 'Sales Order Has Unmapped Product(s)';
                    $ledger->last_reconciliation_attempted_at = Carbon::now();
                    $ledger->save();
                    return; // Halt audit trail processing for this product
                }

                if (! $salesOrderFulfillment = $this->salesOrderFulfillments->getSalesOrderFulfillmentForDate(
                    $salesOrder,
                    Carbon::parse($ledger->event_datetime)
                )
                ) {
                    /*
                     * We also have fulfillFBA() From the FulfillSalesOrderService, but this is old logic and
                     * may be for multi-channel fulfillment... and is not be fully tested.  For now, we are
                     * just going to write the code here specific to creating an FBA Fulfillment
                     */

                    $salesOrderFulfillment = new SalesOrderFulfillment([
                        'sales_order_id' => $salesOrder->id,
                        'fulfillment_type' => SalesOrderFulfillment::TYPE_FBA,
                        'fulfilled_shipping_method' => $detailReport->carrier,
                        'status' => SalesOrderFulfillment::STATUS_FULFILLED,
                        'tracking_number' => $detailReport->tracking_number,
                        'warehouse_id' => $ledger->integrationInstance->warehouse->id,
                        'fulfilled_at' => Carbon::parse($ledger->event_datetime)->format('Y-m-d H:i:s'),
                    ]);

                    $salesOrderFulfillment->save();
                }

                $salesOrderFulfillmentLine = new SalesOrderFulfillmentLine();
                $salesOrderFulfillmentLine->sales_order_line_id = $salesOrderLine->id;
                $salesOrderFulfillmentLine->quantity = -$ledger->quantity;
                $salesOrderFulfillment->salesOrderFulfillmentLines()->save($salesOrderFulfillmentLine);
                $salesOrderLine->fulfilled_quantity += $salesOrderFulfillmentLine->quantity;
                $salesOrderLine->save();

                $inventoryManager = (new InventoryManager($salesOrderLine->warehouse_id, $salesOrderLine->product));
                // Create reservation, then fulfill
                $inventoryManager->takeFromStock($salesOrderFulfillmentLine->quantity, $salesOrderLine, true, null, InventoryDeficiencyActionType::ACTION_TYPE_EXCEPTION, Carbon::parse($ledger->event_datetime));
                $salesOrderFulfillmentLine->reverseReservedInventoryMovements($salesOrderFulfillmentLine->quantity);

                (new ProductInventoryManager(Arr::wrap($product->id)))->setUpdateAverageCost(false)->updateProductInventoryAndAvgCost();
            }

            // Linking the Sales Fulfillment Line to the Ledger Report
            $ledger->skuLink()->associate($salesOrderFulfillmentLine);
            $ledger->last_reconciliation_attempted_at = Carbon::now();
            $ledger->reconciled_at = Carbon::now();
            $ledger->errorLog = null;
            $ledger->blocked_by_ledger_id = null;
            $ledger->save();

            $ledger->amazonFnskuProduct->reconciled_quantity += $ledger->quantity;
            $ledger->amazonFnskuProduct->save();
        } else {
            customlog('amazon', 'Sales order for Shipment Ledger (ID: '.$ledger->id.') could not be determined due to no detail report match.');
            $notes = "Sales order for Shipment Ledger (ID: $ledger->id) could not be determined due to no detail report match.";
            app(AmazonCreateInventoryAdjustmentFromLedger::class)->handle($ledger, $notes);
        }
    }
}