<?php

namespace Modules\Amazon\Actions\LedgerActions\SkuModels;

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelOrder;
use App\Collections\SalesCreditLineCollection;
use App\DTO\SalesCreditDto;
use App\DTO\SalesCreditLineDto;
use App\Exceptions\InsufficientStockException;
use App\Models\SalesCredit;
use App\Models\SalesCreditReturn;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Repositories\SalesCreditRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Arr;
use Modules\Amazon\Abstractions\AmazonLedgerHandlerInterface;
use Modules\Amazon\Actions\LedgerActions\AmazonCreateInventoryAdjustmentFromLedger;
use Modules\Amazon\Entities\AmazonFbaReportCustomerReturn;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Managers\AmazonOrderManager;
use Modules\Amazon\Repositories\AmazonOrderItemRepository;
use Modules\Amazon\Repositories\AmazonOrderRepository;
use Throwable;

/*
 * A Sales Credit must be created, but the sales order associated to it should exist and can be found
 * using the detail report AmazonFbaReportCustomerReturns.order_id
 *
 * If no detail report exists (despite having been requested), can create an adjustment instead
 */
class AmazonProcessSalesCreditReturnLine implements AmazonLedgerHandlerInterface
{
    public function __construct(
        private readonly AmazonOrderRepository $orders,
        private readonly AmazonOrderItemRepository $orderItems,
        private readonly SalesCreditRepository $salesCredits,
    ) {}

    /**
     * @throws Throwable
     */
    public function handle(AmazonFbaReportInventoryLedger $ledger): void
    {
        if ($detailReport = $ledger->details?->first()) {
            $this->processDetailReport($ledger, $detailReport);
        } else {
            $this->handleMissingDetailReport($ledger);
        }
    }

    /**
     * @throws Throwable
     */
    private function processDetailReport(
        AmazonFbaReportInventoryLedger $ledger,
        AmazonFbaReportCustomerReturn $detailReport,
    ): void
    {
        if (!$amazonOrder = $this->getAmazonOrder($ledger, $detailReport)) {
            return;
        }
        if (!$salesOrder = $this->getSalesOrder($ledger, $amazonOrder)) {
            return;
        }
        $salesOrderLine = $this->getSalesOrderLine($ledger, $amazonOrder);
        if (!$this->hasFulfillments($ledger, $salesOrder, $amazonOrder)) {
            return;
        };
        $this->processSalesCreditAndReturn($ledger, $salesOrder, $salesOrderLine);
    }

    private function processSalesCreditAndReturn(AmazonFbaReportInventoryLedger $ledger, SalesOrder $salesOrder, SalesOrderLine $salesOrderLine): void
    {
        $salesCredit = $this->createSalesCredit($ledger, $salesOrder, $salesOrderLine);
        $ledger = $this->createSalesCreditReturn($ledger, $salesCredit);
        $this->updateLedger($ledger);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    private function getAmazonOrder(AmazonFbaReportInventoryLedger $ledger, AmazonFbaReportCustomerReturn $detailReport): null|AmazonOrder|AbstractSalesChannelOrder
    {
        if (! $amazonOrder = $this->orders->getOrderFromId($ledger->integrationInstance, $detailReport->order_id)) {
            customlog('amazon', 'No Amazon Order found for ID: '.$detailReport->order_id);
            /*
             * If no Amazon Order exists, we can't create a Sales Credit.  But the Amazon order will never be downloaded.
             * The reason is that Amazon Shipments get processed before Customer Returns, so the only way to get to
             * this point is if the Amazon Shipment happened prior to the Inventory Start Date, in which case there should just be an adjustment.
             */
            $notes = "Customer Return Ledger (ID: $ledger->id) does not have a corresponding fulfillment, so processing as an adjustment instead of Sales Credit Return.  This is likely due to a Return on a Sales Order that occurred prior to Initial Count.";
            app(AmazonCreateInventoryAdjustmentFromLedger::class)->handle($ledger, $notes);
        }

        return $amazonOrder;
    }

    /**
     * @throws Throwable
     */
    private function getSalesOrder(AmazonFbaReportInventoryLedger $ledger, AmazonOrder $amazonOrder): ?SalesOrder
    {
        if (! $salesOrder = $amazonOrder->salesOrder) {
            customlog('amazon', 'No Sales Order found for Amazon Order ID: '.$amazonOrder->id.', attempting to create...');
            (new AmazonOrderManager($ledger->integrationInstance))->createSkuOrders(Arr::wrap($amazonOrder->id));
            $amazonOrder->refresh();
            if (! $salesOrder = $amazonOrder->salesOrder) {
                customlog('amazon', 'No Sales Order found for Amazon Order ID: '.$amazonOrder->id);
                $ledger->errorLog = 'Missing Sales Order';
                $ledger->last_reconciliation_attempted_at = Carbon::now();
                $ledger->save();
            }
        }

        return $salesOrder;
    }

    /**
     * @throws Throwable
     */
    private function getSalesOrderLine(AmazonFbaReportInventoryLedger $ledger, AmazonOrder $amazonOrder): SalesOrderLine
    {
        $amazonOrderItem = $this->orderItems->getFromAsin($amazonOrder, $ledger->asin);

        if (! $salesOrderLine = $amazonOrderItem->salesOrderLine) {
            // We want to throw an exception here to analyze how this could happen before deciding to handle it with an adjustment
            throw new Exception('Sales order for Customer Return Ledger (ID: '.$ledger->id.') could not be determined due to no sales order line match.');
        }

        return $salesOrderLine;
    }

    /**
     * @throws Throwable
     * @throws InsufficientStockException
     */
    private function hasFulfillments(AmazonFbaReportInventoryLedger $ledger, SalesOrder $salesOrder, AmazonOrder $amazonOrder): bool
    {
        if (! $salesOrder->salesOrderFulfillments->count() > 0) {
            if ($amazonOrder->PurchaseDateUtc->lt($ledger->integrationInstance->fbaInventoryTrackingStartDate())) {
                customlog('amazon', 'Customer Return Ledger (ID: '.$ledger->id.') does not have a corresponding fulfillment, so processing as an adjustment instead of Sales Credit Return.  This is likely due to a Return on a Sales Order that occurred prior to Initial Count.');
                $notes = "Customer Return Ledger (ID: $ledger->id) does not have a corresponding fulfillment, so processing as an adjustment instead of Sales Credit Return.  This is likely due to a Return on a Sales Order that occurred prior to Initial Count.";
                app(AmazonCreateInventoryAdjustmentFromLedger::class)->handle($ledger, $notes);
                return false;
            }
            customlog('amazon', 'Customer Return Ledger (ID: '.$ledger->id.') does not have a corresponding fulfillment, so processing as an adjustment instead of Sales Credit Return.  This is likely due to bad Amazon data causing the fulfillment to be missing a corresponding ledger.');
            // Bad amazon data means we just have to do the adjustment here
            $notes = "Customer Return Ledger (ID: $ledger->id) does not have a corresponding fulfillment, so processing as an adjustment instead of Sales Credit Return.  This is likely due to bad Amazon data causing the fulfillment to be missing a corresponding ledger.";
            app(AmazonCreateInventoryAdjustmentFromLedger::class)->handle($ledger, $notes);
            return false;
        }
        return true;
    }

    private function createSalesCredit(AmazonFbaReportInventoryLedger $ledger, SalesOrder $salesOrder, SalesOrderLine $salesOrderLine): SalesCredit
    {
        $salesCreditLineDto = SalesCreditLineDto::from([
            'description' => $salesOrderLine->description,
            'product_id' => $salesOrderLine->product_id,
            'sales_order_line_id' => $salesOrderLine->id,
            'quantity' => $ledger->quantity,
            'amount' => $salesOrderLine->amount,
            'unit_cost' => $salesOrderLine->salesOrderLineFinancial?->cogs ?? 0,
        ]);

        $salesCreditDto = SalesCreditDto::from([
            'sales_credit_number' => 'CR-'.$salesOrder->sales_order_number,
            'sales_order_id' => $salesOrder->id,
            'credit_date' => Carbon::parse($ledger->event_datetime),
            'to_warehouse_id' => $ledger->integrationInstance->warehouse->id,
            'sales_credit_note' => 'Amazon FBA Customer Return',
            'sales_credit_lines' => new SalesCreditLineCollection([$salesCreditLineDto]),
        ]);

        return $this->salesCredits->create($salesCreditDto);
    }

    private function createSalesCreditReturn(AmazonFbaReportInventoryLedger $ledger, SalesCredit $salesCredit): AmazonFbaReportInventoryLedger
    {
        $detailReport = $ledger->details->first();

        $this->salesCredits->receiveAll(
            $salesCredit,
            SalesCreditReturnLine::ACTION_ADD_TO_STOCK,
            $detailReport->reason,
            $detailReport->customer_comments ?? '',
        );

        /** @var SalesCreditReturn $salesCreditReturn */
        $salesCreditReturn = $salesCredit->salesCreditReturns()->first();

        /** @var SalesCreditReturnLine $salesCreditReturnLine */
        $salesCreditReturnLine = $salesCreditReturn->salesCreditReturnLines()->first();

        // Linking the Sales Credit Return Line to the Ledger Report
        $ledger->skuLink()->associate($salesCreditReturnLine);

        return $ledger;
    }

    private function updateLedger(AmazonFbaReportInventoryLedger $ledger): void
    {
        $ledger->errorLog = null;
        $ledger->last_reconciliation_attempted_at = Carbon::now();
        $ledger->reconciled_at = Carbon::now();
        $ledger->blocked_by_ledger_id = null;
        $ledger->save();
        $ledger->amazonFnskuProduct->reconciled_quantity += $ledger->quantity;
        $ledger->amazonFnskuProduct->save();
    }

    private function handleMissingDetailReport(AmazonFbaReportInventoryLedger $ledger): void
    {
        customlog('amazon', 'Sales order for Customer Return Ledger (ID: '.$ledger->id.') could not be determined due to no detail report match.');
        $ledger->errorLog = 'Pending Detail Report Processing';
        $ledger->last_reconciliation_attempted_at = Carbon::now();
        $ledger->save();
    }
}