<?php

namespace Modules\Amazon\Actions;

use Carbon\Carbon;
use Illuminate\Support\Collection;
use Modules\Amazon\Data\RequestAmazonReportData;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Enums\Entities\ReportProcessingStatusEnum;
use Modules\Amazon\Repositories\AmazonReportRepository;

class GenerateAmazonGapReportRequests
{
    private AmazonReportRepository $reports;
    private Collection $reportRequests;

    public function __construct(
        public AmazonIntegrationInstance $amazonIntegrationInstance,
        public AmazonReportTypeEnum $reportType,
    ) {
        $this->reports = app(AmazonReportRepository::class);
        $this->reportRequests = collect();
    }

    public function __invoke(): Collection
    {
        $existingReports = $this->reports->getReportsForTypeAndStatus(
            $this->amazonIntegrationInstance,
            $this->reportType,
            ReportProcessingStatusEnum::DONE
        );

        $existingReports = $existingReports->sortBy('dataStartTime');

        // Detect gaps between the reports

        $previousReport = null;

        $existingReports->each(function (AmazonReport $report) use (&$previousReport) {
            if ($previousReport && $previousReport->dataEndTime->lt($report->dataStartTime->copy()->subSecond())) {
                $this->createGapReport($previousReport->dataEndTime->copy()->addSecond(), $report->dataStartTime->copy()->subSecond());
            }
            $previousReport = $report;
        });

        return $this->reportRequests;
    }

    private function createGapReport(Carbon $startTime, Carbon $endTime): void
    {
        $this->reportRequests->push(RequestAmazonReportData::from([
            'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
            'report_type' => $this->reportType,
            'data_start_date' => $startTime,
            'data_end_date' => $endTime,
            'createImmediately' => false,
        ]));
    }
}