<?php

namespace Modules\Amazon\Actions\Finance;

use App\Data\FinancialLineData;
use App\Enums\FinancialLineClassificationEnum;
use App\Enums\FinancialLineProrationStrategyEnum;
use App\Models\SalesOrderLine;
use App\Repositories\FinancialLineRepository;
use Illuminate\Support\Collection;
use Modules\Amazon\Entities\AmazonFinancialEvent;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Entities\AmazonOrderItem;
use Modules\Amazon\Repositories\AmazonFinancialEventTypeNominalCodeMappingRepository;

class TransformAmazonFinancialEventToSkuFinancialLines
{
    private AmazonOrder $amazonOrder;

    public function __construct(
        private readonly AmazonFinancialEvent $amazonFinancialEvent
    ) {
        $this->amazonOrder = $this->amazonFinancialEvent->amazonOrder;
    }

    public function handle(): Collection
    {
        $financialLines = collect();
        $amazonNominalCodeMappings = app(AmazonFinancialEventTypeNominalCodeMappingRepository::class)->getAllNominalCodeMappings($this->amazonFinancialEvent->integrationInstance);

        $this->amazonOrder->orderItems->each(function (AmazonOrderItem $amazonOrderItem) use (&$financialLines, $amazonNominalCodeMappings) {
            $amazonFinancialLines = $this->amazonFinancialEvent->getAggregatedFinancialLines($amazonOrderItem);
            $salesOrderLine = $this->amazonOrder->salesOrder->salesOrderLines->where('sales_channel_line_id', $amazonOrderItem->OrderItemId)->first();

            collect($amazonFinancialLines)->each(function ($amount, $chargeType) use (&$financialLines, $salesOrderLine, $amazonNominalCodeMappings) {
                $chargeType = 'Amazon '.$chargeType;
                $amazonNominalCodeMapping = $amazonNominalCodeMappings->where('name', $chargeType)->first();

                if (is_null($amazonNominalCodeMapping)) {
                    $amazonNominalCodeMapping = app(AmazonFinancialEventTypeNominalCodeMappingRepository::class)->firstOrCreate($this->amazonFinancialEvent->integrationInstance, $chargeType);
                }

                $financialLines->push(FinancialLineData::from([
                    'sales_order_id' => $this->amazonOrder->salesOrder->id,
                    'financial_line_type_id' => app(FinancialLineRepository::class)
                        ->getOrCreateFinancialLineType(
                            'Amazon '.$chargeType,
                            FinancialLineClassificationEnum::COST,
                            prorationStrategy: FinancialLineProrationStrategyEnum::REVENUE_BASED,
                            allocateToProducts: true,
                            nominalCodeId: $amazonNominalCodeMapping?->nominalCode?->id
                        )->id,
                    'nominal_code_id' => $amazonNominalCodeMapping?->nominalCode?->id,
                    'description' => 'Amazon '.$chargeType,
                    'quantity' => $salesOrderLine->quantity,
                    'amount' => -$amount,
                    'tax_allocation' => 0,
                    'allocate_to_products' => 1,
                    'allocate_to_id' => $salesOrderLine->id,
                    'allocate_to_type' => SalesOrderLine::class,
                    'proration_strategy' => FinancialLineProrationStrategyEnum::SPECIFIC_LINE,
                    'external_source_id' => $this->amazonFinancialEvent->id,
                    'external_source_type' => get_class($this->amazonFinancialEvent),
                ]));
            });
        });

        return $financialLines;
    }
}
