<?php

namespace Modules\Amazon\Actions;

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelManager;
use Exception;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Entities\AmazonReportRequest;
use Modules\Amazon\Repositories\AmazonReportRepository;
use Modules\Amazon\Services\AmazonClient;
use Throwable;

class CreateAmazonReport extends AbstractSalesChannelManager
{
    protected AmazonReportRepository $reportRepository;

    /**
     * @throws Exception
     */
    public function __construct(protected AmazonReportRequest $amazonReportRequest)
    {
        parent::__construct($this->amazonReportRequest->integrationInstance, new AmazonClient($this->amazonReportRequest->integrationInstance));
        $this->reportRepository = app(AmazonReportRepository::class);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function handle(): AmazonReport
    {
        customlog('amazon', 'Creating report '.$this->amazonReportRequest->reportType->value, [
            'dataStartTime' => $this->amazonReportRequest->dataStartTime,
            'dataEndTime' => $this->amazonReportRequest->dataEndTime,
            'options' => $this->amazonReportRequest->options,
            'marketplaceIds' => $this->amazonReportRequest->marketplaceIds,
        ]);

        // Check to make sure an existing report not in progress and not done doesn't overlap the data range unless it is requested by user, in which case
        // when saving records we will enforce checksum uniqueness
        $overlappingReport = $this->reportRepository->getOverlappingReport($this->amazonReportRequest);
        if ($overlappingReport && !$this->amazonReportRequest->requestedByUser)
        {
            throw new Exception('Report ' . $overlappingReport->id . ' already exists for ' . $this->amazonReportRequest->reportType->value .
                ' (' . $this->amazonReportRequest->dataStartTime . ' to ' . $this->amazonReportRequest->dataEndTime .
                ' and was non failing');
        };

        $report = $this->reportRepository->create(
            $this->amazonReportRequest,
            $this->client->createReport(
                $this->amazonReportRequest->reportType,
                $this->amazonReportRequest->dataStartTime ??
                $this->reportRepository->getStartDateForNew(
                    $this->reportRepository->getModelClassFromReportType($this->amazonReportRequest->reportType)
                ),
                $this->amazonReportRequest->dataEndTime,
                $this->amazonReportRequest->options,
                $this->amazonReportRequest->marketplaceIds,
            )
        );

        // After report is created we delete the request record
        $this->amazonReportRequest->delete();

        customlog('amazon', 'Report created for '.$this->amazonReportRequest->reportType->value, [
            'dataStartTime' => $this->amazonReportRequest->dataStartTime,
            'dataEndTime' => $this->amazonReportRequest->dataEndTime,
            'options' => $this->amazonReportRequest->options,
            'marketplaceIds' => $this->amazonReportRequest->marketplaceIds,
        ]);

        return $report;
    }
}
